<?php

namespace App\Services\Dashboard;

use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Location;
use App\Models\LockStatus;
use App\Models\MakeAndModel;
use App\Models\Region;
use App\Repositories\UserRepository;
use Carbon\Carbon;
use Facades\App\Services\Dashboard\InventoryDashboardService;
use Facades\App\Repositories\Dashboard\DashboardRepository;
use Illuminate\Support\Facades\Cache;

class DashboardService
{
    protected $userRepository;

    public function __construct(UserRepository $userRepository)
    {
        $this->userRepository = $userRepository;
    }

    /**
     * Gets the count of assets for dashboard items
     * @return array
     */
    public function getAssetCounts()
    {
        return [
            'total_assets'          => Asset::count(),
            'inactivePersons'       => Asset::assignedAndLoaned()->inActiveUsers()->count(),
            'inactiveLocations'     => Asset::inActiveLocations()->count(),
            'ageOverFourYears'      => Asset::ageOverFourYears()->count(),
            'damagedOverThirtyDays' => Asset::damagedOverThirtyDays()->count(),
            'lost_stolen_asset'     => Asset::lostStolenAssetWithoutDataWipe()->count(),
            'end_of_life_asset'     => Asset::endOfLifeWithoutCertificate()->count(),
            'loaner_due_week'       => DashboardRepository::loanerDueWeek()->count(),
            'loaner_past_due'       => DashboardRepository::loanerPastDue()->count(),
            'retention_due_week'    => DashboardRepository::retentionDueWeek()->count(),
            'retention_past_due'    => DashboardRepository::retentionPastDue()->count(),
            'lease_due_week'        => DashboardRepository::leaseDueWeek()->count(),
            'lease_past_due'        => DashboardRepository::leasePastDue()->count(),
            'used_new_asset'        => DashboardRepository::assetsInStatuses(['brand_new', 'used', 'used_loaner'])->whereNotNull('location_id')->count(),
        ];
    }

    /**
     * Get assets count for Asset health items
     * @return array
     */
    public function getAssetHealthCounts()
    {
        return [
            'assigned_assets_missing_user'  => DashboardRepository::assetsInStatuses(['assigned'])->userNotAssigned()->count(),
            'missing_tech_specs'            => Asset::withoutTechnicalSpec()->count(),
            'expired_warranty_date'         => InventoryDashboardService::expiredWarrantyEndDate()->count(),
            'outdated_os_version'           => InventoryDashboardService::outdatedOSVersion(),
            'duplicate_serial_no'           => InventoryDashboardService::duplicateSerialno(),
            'duplicate_tag'                 => InventoryDashboardService::duplicateTag(),
            'missing_po'                    => InventoryDashboardService::missingPurchaseOrder()->count(),
            'missing_vendor'                => InventoryDashboardService::missingVendor()->count(),
        ];
    }

    /**
     * Get the Assets count in regions
     * @return mixed
     */
    public function getAssetsInRegionCount()
    {
        $regionCount = [];
        $regionTotalAssets = Cache::remember('region-total-assets', 120, function () {
            return DashboardRepository::getRegionTotalAssets()->get();
        });
        $assetsInRegion = $regionTotalAssets->mapWithKeys(function ($item) {
            return [$item['id'] => $item['count']];
        });

        foreach (Region::get() as $region) {
            $regionCount[] = [
                'count' => isset($assetsInRegion[$region->id]) ? $assetsInRegion[$region->id] : 0,
                'name'  => $region->name,
                'id'    => $region->id,
            ];
        }

        return $regionCount;
    }

    /**
     * Get the computer widgets data
     * @return array
     */
    public function getcomputerWidgetdata()
    {
        $computersWidgetData = [];
        $widgetAssetStatus = ["Brand New Computers" => ["brand_new"], "Used Computers" => ["used"]];
        foreach ($widgetAssetStatus as $statusName => $statusSlug) {
            $computersWidgetData[$statusName] = DashboardRepository::getComputerWidgetData($statusSlug);
        }
        return $computersWidgetData;
    }


    /**
     * When clicking on the widgets on dashboard the link opens in a new tab to a page showing the corresponding data.
     * This function is used to apply sufficient filters on report genrator page to list the data properly.
     * @param mixed $status
     * @param mixed $type
     * 
     * @return [type]
     */
    public function getSearch($status, $type, $searchData)
    {
        $search = [];
        $dateFrom = Carbon::now()->startOfWeek()->format('m/d/Y');
        $dateTo = Carbon::now()->endOfWeek()->format('m/d/Y');

        $assetStatus = AssetStatus::getStatusID($status)->first();
        if ($assetStatus) {
            $assetStatusId = $assetStatus->id;
            $search['asset_status_id'] = ['asset_status_id' => $assetStatusId];
        }

        if ($status == 'retain_hold') {
            if ($type == 'due_this_week') {
                $search['loaner_retention_date_from'] = ['loaner_retention_date' => $dateFrom];
                $search['loaner_retention_date_to'] = ['loaner_retention_date' => $dateTo];
            }

            if ($type == 'past_due') {
                $search['loaner_retention_date_to'] = ['loaner_retention_date' => $dateFrom];
            }
        }

        if ($status == 'loaned') {
            if ($type == 'due_this_week') {
                $search['loaner_return_date_from'] = ['loaner_return_date' => $dateFrom];
                $search['loaner_return_date_to'] = ['loaner_return_date' => $dateTo];
            }

            if ($type == 'past_due') {
                $search['loaner_return_date_to'] = ['loaner_return_date' => $dateFrom];
            }
        }

        if ($status == 'lease') {
            if ($type == 'due_this_week') {
                $search['lease_end_date_from'] = ['lease_end_date' => $dateFrom];
                $search['lease_end_date_to'] = ['lease_end_date' => $dateTo];
            }

            if ($type == 'past_due') {
                $search['lease_end_date_to'] = ['lease_end_date' => $dateFrom];
            }
        }

        if ($status == 'usedandnew') {
            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::whereIn('slug', ['brand_new', 'like_new', 'used'])->pluck('id')->toArray()
            ];

            $locations = Location::where('region_id', $type)->pluck('id')->toArray();
            $search['location'] = [
                'location_id' => $locations
            ];
        }

        if ($status == 'asset_type') {
            $assetTypeId = AssetType::getBySlug($type)->first()->id;
            $search['asset_type'] = [
                'asset_type_id' => $assetTypeId
            ];
        }

        if ($status == 'inactive_users') {
            $search['user_status'] = [
                'status' => 2
            ];
            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusID(['assigned', 'loaned'])->pluck('id')->toArray()
            ];
        }

        if ($status == 'inactive_locations') {
            $search['location_status'] = [
                'status' => 2
            ];
        }

        if ($status == 'ageover_fouryears') {
            $search['asset_age_value'] = [
                'asset_age_value' => 4,
            ];
            $search['asset_age_unit'] = [
                'asset_age_unit' => 'year'
            ];
        }

        if ($status == 'damagedover_thirtydays') {
            $dateTo =  Carbon::now()->subDays(30)->format('m/d/Y');
          
            $search['updated_at'] = [
                $search['last_status_update_to'] = ['last_status_update' => $dateTo]
            ];
        
            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusID('damaged')->first()->id
            ];
        }

        if ($type == 'without_data_wipe') {
            $search['wipe_confirmation'] = [
                'wipe_confirmation' => 2
            ];
        }

        if ($type == 'without_certificate' || $searchData == 'without_certificate') {
            $search['end_of_life_data_wipe_certificate'] = [
                'ewaste_certificate' => 2,
            ];
        }
        if ($status == 'assigned_missinguser') {
            $search['user'] = [
                'user_id' => 'null',
            ];

            $search['asset_status_id'] = [
                'asset_status_id' => AssetStatus::getStatusID('assigned')->first()->id,
            ];
        }

        if ($status == 'missingtechspec') {
            $search['technical_spec'] = [
                'technical_spec_id' => 'null',
            ];
        }

        if ($status == 'missing_po') {
            $search['po_no'] = [
                'po_id' => 'po_null',
            ];
        }

        if ($status == 'missing_vendor') {
            $search['vendor_id'] = [
                'vendor_id' => 'null',
            ];
        }

        if ($type == "chromebook") {
            $assetTypeIds = AssetType::whereIn('slug', ['desktop', 'laptop', 'computer'])->pluck('id')->toArray();
            $makeModels = MakeAndModel::whereIn('asset_type_id', $assetTypeIds)->where('slug', 'like', '%chromebook%')->pluck('id')->toArray();
            $makeModels = !empty($makeModels) ? $makeModels : 'null';
            $search['make_and_model'] = [
                'make_and_model_id' => $makeModels
            ];
            $search['asset_type'] = [
                'asset_type_id' => $assetTypeIds,
            ];
        } else {

            $assetType = AssetType::getBySlug($type)->first();
            if ($assetType) {
                $assetTypeId = $assetType->id;
                $search['asset_type'] = [
                    'asset_type_id' => $assetTypeId,
                ];
            }
        }

        if ($searchData == 'notLocked') {
            $lockStatusIds = LockStatus::where('slug', '!=', 'lock_confirmed')->pluck('id')->toArray();
        } else {
            $lockStatusIds = LockStatus::where('name', $searchData)->pluck('id')->toArray();
        }

        if (!empty($lockStatusIds)) {
            $search['lock_status_id'] = [
                'lock_status_id' => $lockStatusIds,
            ];
        }

        if ($status == 'warranty_expired') {
            $search['warranty_end_date_to'] = [
                'warranty_end_date' => Carbon::yesterday()->format('m/d/Y'),
            ];
        }
        return $search;
    }
}
