<?php

namespace App\Services\CronJob;

use App\Notifications\LoanerItemDue;
use App\Notifications\LoanerItemOverDue;
use App\Notifications\LoanerItemDueWeek;
use App\Notifications\LeaseEndDateToday;
use App\Notifications\LeaseEndDateNextWeek;
use App\Notifications\LoanerItemOverDueMonth;
use App\Notifications\DamagedItems;
use App\Notifications\InternalAdminReport;
use Carbon\Carbon;
use App\User;
use App\Models\Asset;
use Illuminate\Support\Facades\Notification;
use Crypt;
use Cache;
use App\Models\ActiveAdminUsersLog;

class Reminder
{

	public function __construct()
	{
		$this->adminEmail = config('mail.adminAddress');
	}

	public function LoanerItemDue()
	{
		$dateTommorow = Carbon::now()->addDay()->toDateString();
		$assets = Asset::with('user', 'assetType', 'makeAndModel', 'technicalSpec')->hasStatus('loaned')->whereDate('loaner_return_date', $dateTommorow)->get();
		if (!$assets) {
			return false;
		}

		foreach ($assets as $asset) {
			Notification::send($asset->user, new LoanerItemDue($asset));
		}
	}

	public function LoanerItemOverDue()
	{
		$dateYesterday = Carbon::now()->subDay()->toDateString();
		$assets = Asset::with('user', 'assetType', 'makeAndModel', 'technicalSpec')->hasStatus('loaned')->whereDate('loaner_return_date', '<=', $dateYesterday)->get();
		if (!$assets) {
			return false;
		}

		foreach ($assets as $asset) {
			Notification::send($asset->user, new LoanerItemOverDue($asset));
		}
	}

	public function LoanerItemDueWeek()
	{
		$assets = Asset::with('user', 'assetType', 'makeAndModel', 'technicalSpec')->assetDueWeek()->get();

		$adminUser = User::where('email', $this->adminEmail)->first();

		Notification::send($adminUser, new LoanerItemDueWeek($assets));
	}

	public function LeaseEndDateToday()
	{
		$assets = Asset::with('location', 'assetType', 'makeAndModel', 'technicalSpec')->serverAsset()->whereDate('lease_end_date', Carbon::today()->toDateString())->get();
		if (!$assets) {
			return false;
		}

		$adminUser = User::where('email', $this->adminEmail)->first();

		Notification::send($adminUser, new LeaseEndDateToday($assets));
	}

	public function LeaseEndDateNextWeek()
	{
		$assets = Asset::with('location', 'assetType', 'makeAndModel', 'technicalSpec')->serverAsset()->whereDate('lease_end_date', Carbon::today()->addWeek()->toDateString())->get();
		if (!$assets) {
			return false;
		}

		$adminUser = User::where('email', $this->adminEmail)->first();

		Notification::send($adminUser, new LeaseEndDateNextWeek($assets));
	}

	public function DamagedItems()
	{
		$dateLastMonth = Carbon::now()->subDays(30);
		$assets = Asset::with('location', 'assetType', 'makeAndModel', 'technicalSpec', 'latestAssetHistory')->hasStatus('damaged')->whereHas('assetHistory', function ($query) use ($dateLastMonth) {
			$query->where('new_value', 'Damaged')
				->whereDate('created_at', '<=', $dateLastMonth);
		})->get();

		$adminUser = User::where('email', $this->adminEmail)->first();

		Notification::send($adminUser, new DamagedItems($assets));
	}

	public function LoanerItemOverDueMonth()
	{
		$dateLastMonth = Carbon::now()->subDays(30)->toDateString();
		$assets = Asset::with('user', 'assetType', 'makeAndModel', 'technicalSpec')->hasStatus('loaned')->whereDate('loaner_return_date', '<=', $dateLastMonth)->get();
		if (!$assets) {
			return false;
		}

		$adminUser = User::where('email', $this->adminEmail)->first();

		Notification::send($adminUser, new LoanerItemOverDueMonth($assets));
	}

	public function InternalAdminReport()
	{
		$users = [];
		$start = new Carbon('first day of last month');
		$startDate = $start->format('Y-m-d');
		$end = new Carbon('last day of last month');
		$endDate = $end->format('Y-m-d');
		$users['month'] = $start->format('F');
		$beginUsers = ActiveAdminUsersLog::whereDate('created_at', $startDate)->first();
		$endUsers   = ActiveAdminUsersLog::whereDate('created_at', $endDate)->first();
		$users['usersAtBeginning'] = $beginUsers ? $beginUsers->no_of_users : 0;
		$users['usersAtEnd'] = $endUsers ? $endUsers->no_of_users : 0;

		Notification::route('mail', $this->adminEmail)->notify(new InternalAdminReport($users));
	}

	public function refreshNotification()
	{
		Cache::forget('notification8');
	}

	public function activeAdminLog()
	{
		$today = Carbon::now()->toDateString();
		$userCount = User::where('user_type_id', 1)->active()->count();
		$logData = [
			'no_of_users' => $userCount
		];
		ActiveAdminUsersLog::create($logData);
	}
}
