<?php

namespace App\Services\CloudAssets;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CloudAssets\CloudAssetRepository;
use App\Repositories\CommonFilterRepository;
use App\Services\Reports\AbstractReportService;

class CloudAssetService extends AbstractReportService
{
    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Constructor for the class.
     *
     * @param CloudAssetRepository $repository description
     * @param CommonFilterRepository $commonFilterRepository description
     * @param ReportOutputData $reportOutputData description
     */
    public function __construct(CloudAssetRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;

    }

    /**
     * Retrieves data and returns the cloud assets and count.
     *
     * @return array
     */
    public function data()
    {
        $inputData = $this->getInputData();
        $assets = $this->repository->getData();
        $assets = $this->filterWithInputData($assets, $inputData);
        $count  = $assets->count();
        $cloudAssets = $this->reportOutputData->getOutputData($assets);

        return compact('cloudAssets', 'count');
    }

    /**
     * Filter the query results based on the filters applied
     *
     * @param $query
     * @param $inputData
     * @return Query
     */
    public function filterWithInputData($query, $inputData)
    {
        $query = $this->commonFilterRepository->searchToMultipleFields($query, ['name'], $inputData['name']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'identifier', $inputData['identifier']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'availability_zone', $inputData['availability_zone']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'instance_state', $inputData['instance_state']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'cloud_asset_type', $inputData['cloud_asset_type']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'instance_type', $inputData['instance_type']);
        $query = $this->commonFilterRepository->filterWithDirectDateRange($query, 'launched_date', $inputData['date_from'], $inputData['date_to']);

        return $query;
    }

    /**
     * Retrieves the input data from the request.
     *
     * @return array
     */
    public function getInputData(): array
    {
        $requestedData = request()->form ?? request()->all();

        return [
            'name' => $requestedData['name'] ?? '',
            'identifier' => $requestedData['identifier'] ?? '',
            'instance_state' => $requestedData['instance_state'] ?? '',
            'availability_zone' => $requestedData['availability_zone'] ?? '',
            'date_from' => $requestedData['date_from'] ?? '',
            'date_to' => $requestedData['date_to'] ?? '',
            'cloud_asset_type' => $requestedData['cloud_asset_type'] ?? '',
            'instance_type' => $requestedData['instance_type'] ?? '',
        ];
    }

    /**
     * Get the nested data for the report.
     *
     * @param datatype $asset The asset to generate the report for
     * @return array The nested data for the report
     */
    public function getReportNestedData($asset)
    {
        $nestedData = [];
        $nestedData['report']['name']               = $nestedData['export']['Name']             = disableCSVInjection($asset->name);
        $nestedData['report']['cloud_asset_type']   = $nestedData['export']['Cloud Asset Type'] = disableCSVInjection(config("aws-cloud-assets.cloud_asset_type.{$asset->cloud_asset_type}"));
        $nestedData['report']['identifier']         = $nestedData['export']['Identifier']       = disableCSVInjection($asset->identifier);
        $nestedData['report']['instance_state']     = $nestedData['export']['State']   = disableCSVInjection($asset->instance_state);
        $nestedData['report']['instance_type']      = $nestedData['export']['Type']    = disableCSVInjection($asset->instance_type);
        $nestedData['report']['availability_zone']  = $nestedData['export']['Region'] = disableCSVInjection(config("aws-cloud-assets.regions.{$asset->availability_zone}"));
        $nestedData['report']['launched_date']      = $nestedData['export']['Launched Date']    = disableCSVInjection($asset->launched_date);

        return $nestedData;
    }

    /**
     * Retrieves the nested data for a given item.
     *
     * @param mixed $item The item for which to retrieve the nested data.
     * @param int $index The index of the item.
     * @return array The nested data for the item.
     */
    public function getNestedData($item, $index)
    {
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id'] = $index;

        return $nestedData;
    }

    /**
     * Generating data table data from query result
     * @param $query Query Result
     * @param $start Starting index
     * @param $data Collection variable that will return
     *
     * @return array
     */
    public function getReportData($query, $start, $data = [])
    {
        $parentIndex = $start;

        foreach ($query as $key => $singleRecord) {
            $parentIndex++;
            $nestedData = $this->getNestedData($singleRecord, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Export the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $requestedData = request()->all();
        $inputData = $this->getInputData($requestedData);
        $assets = $this->repository->getData();
        $assets = $this->filterWithInputData($assets, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }

    /**
     * Get the search field values from the request and repository.
     *
     * @return array
     */
    public function getFilterFieldValues()
    {
        $filed = request('field');
        $searchText = request('search');
        $res = $this->repository->getFilterFieldData($filed, $searchText);

        $result = $res->mapWithKeys(function ($item, $key) use ($filed) {
            return [
                $key => [
                    'id' => $item[$filed],
                    'text' => $item[$filed],
                ]
            ];
        });

        return $result;
    }
}
?>