<?php

namespace App\Services\BulkUpload;

use Facades\App\Services\AssetHistory as AssetHistoryService;
use App\Events\BulkUpdates;
use Illuminate\Support\Facades\Auth;
use App\Models\Location;
use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use Carbon\Carbon;
use App\Services\Integrations\Tickets\TicketManagementService;

class SpecialBulkLinkService extends SpecialBulkAbstract
{

    /**
     * Class constructor.
     *
     * @param TicketManagementService $ticketManagementService The ticket management service instance.
     */
    public function __construct(protected TicketManagementService $ticketManagementService)
    {
    }

    /**
     * Retrieves the header map configuration for bulk uploads.
     *
     * @return array The header map configuration.
     */
    public function getHeaderMap()
    {
        return config('bulk-upload.linkData');
    }

    /**
     * Imports asset data from a given file path.
     *
     * @param string      $path     The path to the file containing asset data.
     * @param string|null $fileName The optional name of the file.
     *
     * @return array An array of error data encountered during import.
     */
    public function importAssetData(string $path, $fileName = null)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $errorData = [];
        $ticketNo = '';
        $errorData['validationErrors'] = $this->validateFields($items);

        if (!empty($items) && empty($errorData['validationErrors'])) {
            unset($errorData['validationErrors']);

            foreach ($items as $item) {
                if (!count($item)) {
                    continue;
                }

                $csvData = $this->generateCSVData($item);
                $assetData = $csvData['assetData'];

                if ($csvData['errorData']) {
                    $errorData[] = $csvData['errorData'];

                    continue;
                }

                if (isset($assetData['parent_asset_id']) && isset($assetData['child_asset_id'])) {
                    $this->saveData($assetData);
                }

                $ticketNo = $ticketNo ? $ticketNo : $assetData['ticket_no'];
            }

            $this->attachFiletoTicket($ticketNo, $path, $fileName);
        }

        return $errorData;
    }

    /**
     * Generates CSV data from the given item.
     *
     * @param array $item The item to process.
     *
     * @return array An array containing 'assetData' and 'errorData'.
     */
    public function generateCSVData(array $item)
    {
        if (empty($item)) {
            return false;
        }

        $assetData    = [];
        $errorData    = '';

        foreach ($this->getHeaderMap() as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }

            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == 'parent_serial_no') {
                $assetData = $this->getAdditionalLinkData($assetData, $item[$fileHeader], 'parent');
            }

            if ($dbField == 'child_serial_no') {
                $assetData = $this->getAdditionalLinkData($assetData, $item[$fileHeader], 'child');
            }

            if (isset($item[$dbField]) && !$item[$dbField]) {
                $errorData .= 'The ' . $item[$fileHeader] . ' do not exist.' . "\r\n";
            }
        }

        $errorData .= $this->canBeLinked($assetData);

        return compact('assetData', 'errorData');
    }

    /**
     * Checks if the given assets can be linked and gathers error information.
     *
     * @param array $csvData The CSV data containing asset information.
     *
     * @return string A string containing any error messages related to linking the assets.
     */
    public function canBeLinked(array $csvData)
    {
        $errors = '';
        $data = $csvData;

        if (isset($data['parent_asset_id'])) {
            if (Asset::where('serial_no', $data['parent_serial_no'])->parent()->count() < 1) {
                $errors .= 'Attempting to link to an asset, which is not a parent.' . "\r\n";
            }

            if (!Asset::find($data['parent_asset_id'])->canBeLinked()) {
                $errors .= 'Parent asset can not link due to status restrictions.' . "\r\n";
            }
        }

        if (isset($data['child_asset_id'])) {
            if (Asset::where('serial_no', $data['child_serial_no'])->child()->count() < 1) {
                $errors .= 'Attempting to link an asset, which is not a child.' . "\r\n";
            }

            if (!Asset::find($data['child_asset_id'])->canBeLinked()) {
                $errors .= 'Child asset can not link due to status restrictions.' . "\r\n";
            }
        }

        return $errors;
    }

    /**
     * Retrieves additional link data for an asset based on its serial number and type.
     *
     * @param array  $assetData The existing asset data.
     * @param string $item      The serial number of the asset.
     * @param string $type      The type of the asset, either 'parent' or 'child'.
     *
     * @return array The updated asset data with additional link information.
     */
    public function getAdditionalLinkData(array $assetData, string $item = '', string $type = 'parent')
    {
        $assetData[$type . '_asset_id'] = null;
        $assetData[$type . '_current_asset_status_id'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();

        if ($asset) {
            $assetData[$type . '_asset_id'] = $asset->id;
            $assetData[$type . '_current_asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    /**
     * Saves data by updating an asset and its parent asset, then adding the asset history.
     *
     * @param array $inputData The input data containing information about the child and parent assets.
     *
     * @return void
     */
    public function saveData(array $inputData)
    {
        $asset = Asset::find($inputData['child_asset_id']);
        $parentAsset = Asset::find($inputData['parent_asset_id']);

        $this->updateAsset($asset, $parentAsset);
        $this->addAssetHistory($asset, $parentAsset, $inputData);
    }

    /**
     * Adds history records for the asset and its parent asset.
     *
     * This method logs the linking of a child asset to a parent asset, including changes to relevant fields.
     *
     * @param Asset $asset       The child asset.
     * @param Asset $parentAsset The parent asset.
     * @param array $inputData   The input data containing additional information for history.
     *
     * @return void
     */
    public function addAssetHistory(Asset $asset, Asset $parentAsset, array $inputData)
    {
        $description = __('history.Linked', [
            'parentAssetId' => $parentAsset->id,
            'childAssetId' => $asset->id,
            'parentAssetname' => $parentAsset->serial_no,
            'childAssetname' => $asset->serial_no,
            'commenttext' => '',
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $inputData['ticket_no'],
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'linked',
            'comments' => '',
            'old_parent_asset_id' => $asset->parent_asset_id,
            'new_parent_asset_id' => $parentAsset->id,
            'old_location_id' => $asset->location_id,
            'new_location_id' => $parentAsset->location_id,
            'old_user_id' => $asset->user_id,
            'new_user_id' => $parentAsset->user_id,
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $parentAsset->asset_status_id,
            'old_value' => $parentAsset->serial_no,
            'new_value' => $asset->serial_no,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
    }

    /**
     * Updates the attributes of a child asset based on its parent asset.
     *
     * @param Asset $asset       The child asset to update.
     * @param Asset $parentAsset The parent asset providing updated values.
     *
     * @return void
     */
    public function updateAsset(Asset $asset, Asset $parentAsset)
    {
        $carrierId = ($parentAsset->carrier_id != null) ? $parentAsset->carrier_id : $asset->carrier_id;

        $asset->update([
            'parent_asset_id' => $parentAsset->id,
            'user_id' => $parentAsset->user_id,
            'location_id' => $parentAsset->location_id,
            'asset_status_id' => $parentAsset->asset_status_id,
            'carrier_id' => $carrierId,
            'linked_date' => Carbon::now(),
        ]);
    }

    /**
     * Retrieves the status slug.
     *
     * @return string The status slug.
     */
    public function getStatusSlug()
    {
        return "";
    }

    /**
     * Retrieves the default ticket configuration for bulk link operations.
     *
     * @return string The default ticket configuration.
     */
    public function getDefaultTicket()
    {
        return config('tickets.bulk_link');
    }

    /**
     * Retrieves specific data from the given asset.
     *
     * This method extracts various fields from the asset and returns them as an array.
     *
     * @param Asset $asset The asset instance from which to retrieve data.
     *
     * @return array The retrieved asset data.
     */
    public function getData(Asset $asset)
    {
        $assetData['id']                 = $asset->id;
        $assetData['asset_status_id']     = null;
        $assetData['location_id']          = $asset->location_id;
        $assetData['loaner_retention_date'] = null;
        $assetData['wipe_confirmation'] = null;
        $assetData['lost_date']         = null;
        $assetData['end_of_life_date']     = null;
        $assetData['ticket_no']         = config('tickets.bulk_link');

        return $assetData;
    }

    /**
     * Attaches a file to the specified ticket.
     *
     * @param mixed  $ticketid The ID of the ticket to which the file will be attached.
     * @param string $filePath The path to the file to be attached.
     * @param string $fileName The name of the file to be attached.
     *
     * @return bool Returns true if the file was successfully attached, false otherwise.
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }

        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk link (special)', Auth::user()->id, $filePath, $fileName);

        return true;
    }
}
