<?php

namespace App\Services\BulkUpload;

use Facades\App\Services\AssetHistory as AssetHistoryService;
use App\Events\BulkUpdates;
use App\Models\AssetStatus;
use App\Models\Asset;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\Services\Integrations\Tickets\TicketManagementService;

class SpecialBulkDamageService extends SpecialBulkAbstract
{
    protected $ticketManagementService;

    public function __construct(TicketManagementService $ticketManagementService)
    {
        $this->ticketManagementService = $ticketManagementService;
    }
    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.damagedAssetData');
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "damaged";
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return Config('tickets.bulk_damage');
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @return array            
     */
    public function getData(Asset $asset)
    {
        $assetData['id']                     = $asset->id;
        $assetData['loaner_retention_date'] = null;
        $assetData['wipe_confirmation']     = null;
        $assetData['lost_date']             = null;
        $assetData['end_of_life_date']         = null;
        $assetData['user_id']                 = null;
        $assetData['ticket_no']             = $this->getDefaultTicket();

        return $assetData;
    }

    /**
     * Save Asset History
     * @param  Asset  $asset
     * @param  array  $data
     * @return void
     */
    public function saveAssetHistory(Asset $asset, array $data)
    {
        $status = optional(AssetStatus::find($data['asset_status_id']))->name;

        $comments = null;
        $description = AssetHistoryService::getStatusDescription($status, $data, $asset, $comments);
        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $data['ticket_no'],
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'status_updated',
            'comments' => $comments,
            'old_value' => $asset->assetStatus->name,
            'new_value' => $status,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        $assetHistory = $this->getAdditionalHistoryData($assetHistory, $data, $asset);
        event(new BulkUpdates($assetHistory));
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk damage (special)', Auth::user()->id, $filePath, $fileName);
        return true;
    }
}
