<?php

namespace App\Services\BulkUpload;

use App\Http\Traits\BulkUploadTrait;
use App\Events\BulkUpdates;
use App\Models\Asset;
use App\Models\AssetStatus;
use Facades\App\Services\AssetHistory as AssetHistoryService;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

abstract class SpecialBulkAbstract
{
    use BulkUploadTrait;

    abstract protected function getHeaderMap();
    abstract protected function getStatusSlug();
    abstract protected function getData(Asset $asset);
    abstract protected function attachFiletoTicket($ticketNo, $path, $fileName);

    /**
     * Imports asset data from a given file path.
     *
     * @param string      $path     The path to the file containing asset data.
     * @param string|null $fileName The optional name of the file.
     *
     * @return array An array of error data encountered during import.
     */
    public function importAssetData(string $path, $fileName = null)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $errorData = [];
        $ticketNo = '';
        $errorData['validationErrors'] = $this->validateFields($items);

        if (!empty($items) && empty($errorData['validationErrors'])) {
            unset($errorData['validationErrors']);

            foreach ($items as $item) {
                if (!count($item)) {
                    continue;
                }

                $csvData = $this->generateCSVData($item);
                $assetData = $csvData['assetData'];

                if ($csvData['errorData']) {
                    $errorData[] = $csvData['errorData'];
                    continue;
                }

                if (isset($assetData['id'])) {
                    $asset = Asset::with('assetStatus', 'childrenAsset')->where('id', $assetData['id'])->first();

                    if ($asset) {
                        unset($assetData['id']);
                        $this->saveAssetHistory($asset, $assetData);
                        $asset->update($assetData);

                        if ($asset->has('childrenAsset')) {
                            unset($assetData['serial_no']);

                            foreach ($asset->childrenAsset as $assetChild) {
                                $this->saveAssetHistory($assetChild, $assetData);
                                $assetChild->update($assetData);
                            }
                        }
                    }
                }

                $ticketNo = $ticketNo ? $ticketNo : $assetData['ticket_no']; //taking the ticket number of first row
            }

            $this->attachFiletoTicket($ticketNo, $path, $fileName);
        }

        return $errorData;
    }

    /**
     * Validates the fields of the given CSV data.
     *
     * @param array $csvData The CSV data to validate.
     *
     * @return array An array of validation errors, if any.
     */
    public function validateFields(array $csvData)
    {
        $errors = [];

        if (!isset($csvData[0])) {
            $errors[] =  'Invalid Template.';

            return $errors;
        }

        foreach (array_values($this->getHeaderMap()) as $key) {
            if (!in_array($key, array_keys($csvData[0]))) {
                $errors[] =  $key . ' is missing in the Template.';
            }
        }

        return $errors;
    }

    /**
     * Generates CSV data from the given item.
     *
     * @param array $item The item to process.
     *
     * @return array An array containing 'assetData' and 'errorData'.
     */
    public function generateCSVData(array $item)
    {
        if (empty($item)) {
            return false;
        }

        $assetData    = [];
        $errorData    = '';

        foreach ($this->getHeaderMap() as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result == false) {
                if ($dbField == "location_id") {
                    $errorData .= 'Location ' . $item[$fileHeader] . ' do not exist.' . "\r\n";
                }

                if ($dbField == "user_id") {
                    $errorData .= 'User with email ' . trim($item[$fileHeader]) . ' do not exist.' . "\r\n";
                }
            }

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }

            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "serial_no") {
                if (!empty($item[$fileHeader])) {
                    $asset = Asset::where('serial_no', trim($item[$fileHeader]))->first();

                    if ($asset) {
                        $assetData = $this->getData($asset);
                    }

                    if (!$asset) {
                        $errorData .= 'Asset with serial # ' . $item[$fileHeader] . ' do not exist.' . "\r\n";
                    }
                }
            }
        }

        return compact('assetData', 'errorData');
    }

    /**
     * Saves the history of the given asset with the provided data.
     *
     * @param Asset $asset The asset for which the history is being saved.
     * @param array $data  The data related to the asset changes.
     *
     * @return void
     */
    public function saveAssetHistory(Asset $asset, array $data)
    {
        $status = AssetStatus::getStatusID($this->getStatusSlug())->first()->name;

        $comments = null;
        $description = AssetHistoryService::getStatusDescription($status, $data, $asset, $comments);

        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $data['ticket_no'],
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'status_updated',
            'comments' => $comments,
            'old_value' => optional($asset->assetStatus)->name,
            'new_value' => $status,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        $assetHistory = $this->getAdditionalHistoryData($assetHistory, $data, $asset);
        event(new BulkUpdates($assetHistory));
    }
}
