<?php

namespace App\Services\BulkUpload;

use App\Models\Asset;
use App\Models\Location;
use App\User;
use App\Services\Asset\UpdateStatusService;
use App\Rules\UserRequiredAssetStatus;
use App\Rules\LocationRequiredAssetStatus;
use App\Models\Vendor;
use App\Rules\BulkUploadDateFormat;
use App\Rules\ValidatePastDate;
use App\Rules\VendorRequiredAssetStatus;
use Illuminate\Support\Facades\Validator;
use Exception;
use App\Services\Integrations\Tickets\TicketManagementService;
use Illuminate\Support\Facades\Auth;

class BulkStatusUpdate extends BulkAbstract
{

    public function __construct(protected UpdateStatusService $updateStatusService, protected TicketManagementService $ticketManagementService) {}

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-status-update-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.updateStatusAssetData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return Config('tickets.bulk_status_update');
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug($data = null)
    {
        if (isset($data['asset_status_id'])) {
            $this->getStatus($data['asset_status_id'])->slug;
        }

        return "";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName($data = null)
    {
        if (isset($data['asset_status_id'])) {
            $this->getStatus($data['asset_status_id'])->name;
        }

        return "";
    }

    /**
     * Generate asset data from the csv array
     * @param  mixed $item
     * @param  int    $count
     * @return array
     */
    public function generateCSVData(array $item, int $count, $path = null, $fileName = null)
    {
        $assetData    = [];
        foreach ($this->getHeaderMap() as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);
            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "serial_no") {
                if (isset($item[$fileHeader]) && !empty($item[$fileHeader])) {
                    $assetData = $this->getAdditionalAssetData($assetData, $item[$fileHeader]);
                }
            }
        }

        $assetData = $this->addAdditionalData($assetData);

        $assetData['asset_count'] = $count++;

        if ($fileName && $path) {
            $assetData['filePath'] = $path;
            $assetData['fileName'] = $fileName;
        }

        session(['asset_count' => $assetData['asset_count']]);

        return compact('assetData');
    }

    /**
     * Get Additonal Asset Data
     * @param  array  $assetData
     * @param  string $item
     * @return array
     */
    public function getAdditionalAssetData(array $assetData, string $item = "")
    {
        $assetData['asset_id'] = null;
        $assetData['current_asset_status_id'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();
        if ($asset) {
            $assetData['asset_id'] = $asset->id;
            $assetData['current_asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    /**
     * Add Additional Data to show in the uploaded view
     * @param array $data
     * @return array
     */
    public function addAdditionalData(array $data)
    {
        /**
         * if csv user / location is null then assign asset user / location based on asset status
         */
        $hasLocation = isset($data['asset_status_id']) ? $this->getStatus($data['asset_status_id'])->has_location : '';
        $hasUser = isset($data['asset_status_id']) ? $this->getStatus($data['asset_status_id'])->has_user : '';

        $asset = isset($data['asset_id']) ? Asset::find($data['asset_id']) : null;
        if ($hasLocation) {
            $data['location_id'] = $data['location_id'] ? $data['location_id'] : ($asset ? $asset->location_id : null);
            $data['location_name'] = isset($data['location_id']) ? Location::find($data['location_id'])->room_name : '';
        } else {
            $data['location_id'] = '';
            $data['location_name'] = '';
        }
        if ($hasUser) {
            $data['user_id'] = $data['user_id'] ? $data['user_id'] : ($asset ? $asset->user_id : null);
            try {
                $assetUser =  User::findOrFail($data['user_id']);
            } catch (Exception $e) {
            }
            $data['user_name'] = isset($assetUser) ? $assetUser->userName : '';
        } else {
            $data['user_id'] = '';
            $data['user_name'] = '';
        }

        /**
         * if csv user / location is null then assign asset user / location end
         */

        $data['current_status_slug'] = isset($data['current_asset_status_id']) ? $this->getStatus($data['current_asset_status_id'])->slug : '';
        $data['current_status_name'] = isset($data['current_asset_status_id']) ? $this->getStatus($data['current_asset_status_id'])->name : '';
        $data['status_slug'] = isset($data['asset_status_id']) ? $this->getStatus($data['asset_status_id'])->slug : '';
        $data['status_name'] = isset($data['asset_status_id']) ? $this->getStatus($data['asset_status_id'])->name : '';
        $data['vendor_name'] = isset($data['vendor_id']) ? Vendor::find($data['vendor_id'])?->name : '';


        return $this->getAdditionalStatusData($data);
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'         => 'required|exists:assets,serial_no',
                'asset_status_id'   => 'required',
                'user_id'           => new UserRequiredAssetStatus($data['asset_status_id'], $count, 1),
                'location_id'       => new LocationRequiredAssetStatus($data['asset_status_id'], $count),
                'loaner_return_date' => new BulkUploadDateFormat($data['loaner_return_date'], config('date.formats.read_date_format'), $count, 'expected return'),
                'lost_date' => [new BulkUploadDateFormat($data['lost_date'], config('date.formats.read_date_format'), $count, 'stolen/lost'), new ValidatePastDate($data['lost_date'], config('date.formats.read_date_format'), $count, 'stolen/lost')],
                'loaner_retention_date' => new BulkUploadDateFormat($data['loaner_retention_date'], config('date.formats.read_date_format'), $count, 'retain hold release'),
                'vendor_id'         => new VendorRequiredAssetStatus($data['asset_status_id'], $count),
                'wipe_confirmation' => ['nullable', 'required_if:current_status_slug,retain_hold'],
                'retain_hold_wipe_confirmed_date' => ['nullable', 'required_if:current_status_slug,retain_hold', 'date_format:' . config('date.formats.read_date_format')],
                'retain_hold_note' => ['nullable', 'required_if:current_status_slug,retain_hold'],
            ],
            [
                'serial_no.required'        => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'          => 'Line no ' . $count . ' : The serial # does not exists',
                'asset_status_id.required'  => 'Line no ' . $count . ' : The Asset Status does not exists',
                'wipe_confirmation.required_if' => 'Line no ' . $count . ' : The retain hold wipe confirmation is required when the current status is Retain Hold.',
                'retain_hold_wipe_confirmed_date.required_if' => 'Line no ' . $count . ' : The retain hold wipe confirmation date is required when the current status is Retain Hold.',
                'retain_hold_wipe_confirmed_date.date_format' => 'Line no ' . $count . ' : The retain hold wipe confirmation date must be in the format ' . config('date.formats.read_date_format') . '.',
                'retain_hold_note.required_if' => 'Line no ' . $count . ' : The retain hold notes is required when the current status is Retain Hold.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Save Data
     * @param  array  $inputData
     * @param  int    $count
     * @return void
     */
    public function saveData(array $inputData, int $count)
    {
        return $this->updateStatusService->updateStatus(json_decode($inputData['asset_id' . $count], true), 'bulk_upload');
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count     
     * @return array            
     */
    public function getData(array $inputData, int $count)
    {
        return [];
    }


    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk Status Update', Auth::user()->id, $filePath, $fileName);

        return true;
    }


    /**
     * Check if the asset can be updated to the status
     * @param  array $csvData
     * @return array
     */
    public function checkCanBeUpdatedToStatus($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        $validatedData = [];

        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $rowCount = $data['asset_count'] + 1;
            $assetType = Asset::with('assetType')->find($data['asset_id']);
            if ($data['asset_status_id']) {
                $canUpdatedStatuses = $this->searchFromStatus($this->getStatus($data['asset_status_id'])->slug);
                $currentStatusSlug = $this->getStatus($data['current_asset_status_id'])->slug;

                if (!in_array($currentStatusSlug, $canUpdatedStatuses)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The serial # : ' . $data['serial_no'] . ' cannot be updated to ' . $this->getStatus($data['asset_status_id'])->name . '.';
                    continue;
                }

                $validatedData[] = $csvData;
            }
        }

        return ['errors' => $errors, 'csvData' => $validatedData];
    }
}
