<?php

namespace App\Services\BulkUpload;

use App\Events\BulkUpdates;
use Illuminate\Support\Facades\Validator;
use App\Models\Asset;
use App\Models\AssetStatus;
use Illuminate\Support\Facades\Auth;
use App\Models\Location;
use App\Rules\StorageLocation;
use App\Services\Asset\AssetStatusService;
use Carbon\Carbon;
use App\Services\Integrations\Tickets\TicketManagementService;

class BulkReturnService extends BulkAbstract
{
    /**
     * Constructs a new instance of the class.
     *
     * @param TicketManagementService $ticketManagementService The TicketManagementService instance used for managing tickets.
     * @param AssetStatusService $assetStatusService The AssetStatusService instance used for retrieving asset status data.
     */
    public function __construct(
        protected TicketManagementService $ticketManagementService,
        protected AssetStatusService $assetStatusService
    ) {}

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-return-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.returnData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return Config('tickets.bulk_return');
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'         => 'required|exists:assets,serial_no',
                'asset_status_id'   => 'required',
                'location_id'       => new StorageLocation($count),
            ],
            [
                'serial_no.required'        => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'          => 'Line no ' . $count . ' : The serial # does not exists',
                'asset_status_id.required'  => 'Line no ' . $count . ' : The asset status is required.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Check if the asset can be returned to the location
     * @param  array $csvData
     * @return array
     */
    public function checkCanBeUpdatedToStatus($csvDatas)
    {
        return $this->checkCanBeReturned($csvDatas);
    }

    /**
     * Checks if the given assets can be returned and gathers error information.
     *
     * @param array $csvDatas The CSV data containing asset information.
     *
     * @return array An array containing any error messages related to returning the assets and
     *      the valid data that can be returned.
     */
    private function checkCanBeReturned(array $csvDatas)
    {
        $errors = [];
        $validData = [];

        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $rowCount = $data['asset_count'] + 1;

            if (isset($data['asset_id'])) {
                $asset = Asset::find($data['asset_id']);

                if (!$asset->canBeReturned()) {
                    $errors[$rowCount] = 'Line no ' . $rowCount . ' : You are attempting to return an asset that is in your inventory.';
                    continue;
                }

                $newStatus = AssetStatus::find($data['asset_status_id']);

                $statusUpdate = $this->assetStatusService->canBeUpdatedTo($newStatus?->slug, $asset);

                if ($statusUpdate === false) {
                    $errors[$rowCount] = 'Line no ' . $rowCount . ' : Sorry, you cannot update the asset status from '
                        . $asset->assetStatus?->name . ' to ' . $newStatus?->name . '.';
                    continue;
                }

                $validData[] = $csvData;
            }
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }


    /**
     * Save Data from the request
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveData(array $inputData, int $i)
    {
        $asset = Asset::find($inputData['asset_id' . $i]);

        if ($asset === null || $asset->canBeReturned() === false) {
            return false;
        }

        $this->saveAssetHistory($asset, $inputData, $i);

        $this->updateAsset($asset, $inputData, $i);

        //While moving the parent asset then move child assets also
        if ($asset->has('childrenAsset')) {

            foreach ($asset->childrenAsset as $assetChild) {

                $this->saveAssetHistory($assetChild, $inputData, $i);

                $this->updateAsset($assetChild, $inputData, $i);
            }
        }

        return true;
    }

    /**
     * Save Asset History
     * @param  Asset  $asset
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveAssetHistory(Asset $asset, array $inputData, int $i = null)
    {
        $commentstext = '';

        $description = __('history.Returned', [
            'assetname' => $asset->serial_no,
            'assetid' => $asset->id,
            'oldroomname' => $this->getOldName($asset),
            'newroomname' => Location::findOrFail($inputData['location_id' . $i])->room_name,
            'oldassetstatus' => $asset->assetStatus->name,
            'newassetstatus' => AssetStatus::find($inputData['asset_status_id' . $i])->name,
            'commenttext' => $commentstext,
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $inputData['ticket_no' . $i] ?? null,
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'returned',
            'comments' => '',
            'old_location_id' => optional($asset->location)->id,
            'new_location_id' => $inputData['location_id' . $i],
            'old_user_id' => $asset->user_id,
            'new_user_id' => NULL,
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $inputData['asset_status_id' . $i],
            'old_value' => $this->getOldName($asset),
            'new_value' => optional(Location::findOrFail($inputData['location_id' . $i]))->room_name,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
    }

    /**
     * Update Asset
     * @param  Asset  $asset
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function updateAsset(Asset $asset, array $inputData, int $i)
    {
        $asset->update([
            'location_id' => $inputData['location_id' . $i],
            'user_id' => NULL,
            'asset_status_id' => $inputData['asset_status_id' . $i],
        ]);
    }

    /**
     * Returns the old name of the asset which is either the room name or the user name based on the asset's location or user.
     *
     * @param Asset $asset
     * @return string
     */
    public function getOldName(Asset $asset)
    {
        if ($asset->location_id) {
            return $asset->location?->room_name;
        }

        if ($asset->user_id) {
            return $asset->user?->user_name;
        }

        return "";
    }

    /**
     * Get Additonal Asset Data
     * @param  array  $assetData
     * @param  string $item
     * @return array
     */
    public function getAdditionalAssetData(array $assetData, string $item = "")
    {
        $assetData['asset_id'] = null;
        $assetData['current_asset_status_id'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();
        if ($asset) {
            $assetData['asset_id'] = $asset->id;
            $assetData['current_asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "";
    }


    /**
     * Get Headers of CSV
     * @return array
     */
    public function getData(array $inputData, int $count)
    {
        return "";
    }


    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk return', Auth::user()->id, $filePath, $fileName);
        return true;
    }
}
