<?php

namespace App\Services\BulkUpload;

use App\Events\BulkUpdates;
use Illuminate\Support\Facades\Validator;
use App\Models\Asset;
use Illuminate\Support\Facades\Auth;
use App\Models\Location;
use Carbon\Carbon;
use App\Services\Integrations\Tickets\TicketManagementService;

class BulkMoveService extends BulkAbstract
{

    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-move-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.moveData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return Config('tickets.bulk_move');
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'        => 'required|exists:assets,serial_no',
                'location_id'    => 'required|exists:locations,id',
                'ticket_no'    => 'nullable|alpha_dash',
            ],
            [
                'serial_no.required'    => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'      => 'Line no ' . $count . ' : The serial # does not exists',
                'location_id.required'  => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'    => 'Line no ' . $count . ' : The location does not exists',
                'ticket_no.required'  => 'Line no ' . $count . ' : The Ticket No is required.',
                'ticket_no.alpha_dash'    => 'Line no ' . $count . ' : Only alpha numeric characters are allowed in the Ticket #',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Check if the asset can be moved to the location
     * @param  array $csvData
     * @return array
     */
    public function checkCanBeUpdatedToStatus($csvDatas)
    {
        return $this->checkCanBeMoved($csvDatas);
    }

    /**
     * Checks if the given assets can be moved and gathers error information.
     *
     * @param array $csvData The CSV data containing asset information.
     *
     * @return array An array containing any error messages related to moving the assets.
     */
    private function checkCanBeMoved(array $csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        $validData = [];

        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $rowCount = $data['asset_count'] + 1;

            if (isset($data['asset_id'])) {
                if (!Asset::find($data['asset_id'])->canBeMoved()) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : You are attempting to move an asset that must be returned to a location first.';
                    continue;
                }

                $validData[] = $csvData;
            }
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }

    /**
     * Save Data from the request
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveData(array $inputData, int $i)
    {
        $asset = Asset::find($inputData['asset_id' . $i]);

        if ($asset === null || $asset->canBeMoved() === false) {
            return false;
        }

        $this->saveAssetHistory($asset, $inputData, $i);

        $this->updateAsset($asset, $inputData, $i);

        //While moving the parent asset then move child assets also
        if ($asset->has('childrenAsset')) {

            foreach ($asset->childrenAsset as $assetChild) {

                $this->saveAssetHistory($assetChild, $inputData, $i);

                $this->updateAsset($assetChild, $inputData, $i);
            }
        }

        return true;
    }

    /**
     * Save Asset History
     * @param  Asset  $asset
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveAssetHistory(Asset $asset, array $inputData, int $i = null)
    {
        $commentstext = '';

        $description = __('history.Moved', [
            'assetname' => $asset->serial_no,
            'assetid' => $asset->id,
            'oldroomname' => optional($asset->location)->room_name,
            'newroomname' => optional(Location::findOrFail($inputData['location_id' . $i]))->room_name,
            'commenttext' => $commentstext,
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $inputData['ticket_no' . $i] ?? null,
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'moved',
            'comments' => '',
            'old_location_id' => optional($asset->location)->id,
            'new_location_id' => $inputData['location_id' . $i],
            'old_value' => optional($asset->location)->room_name,
            'new_value' => optional(Location::findOrFail($inputData['location_id' . $i]))->room_name,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
    }

    /**
     * Get Additonal Asset Data
     * @param  array  $assetData
     * @param  string $item
     * @return array
     */
    public function getAdditionalAssetData(array $assetData, string $item = "")
    {
        $assetData['asset_id'] = null;
        $assetData['current_asset_status_id'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();
        if ($asset) {
            $assetData['asset_id'] = $asset->id;
            $assetData['current_asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    /**
     * Update Asset
     * @param  Asset  $asset
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function updateAsset(Asset $asset, array $inputData, int $i)
    {
        $asset->update([
            'location_id' => $inputData['location_id' . $i],
        ]);
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "";
    }


    /**
     * Get Headers of CSV
     * @return array
     */
    public function getData(array $inputData, int $count)
    {
        return "";
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk move', Auth::user()->id, $filePath, $fileName);

        return true;
    }
}
