<?php

namespace App\Services\BulkUpload;

use Illuminate\Support\Facades\Validator;
use App\Models\AssetStatus;
use App\Rules\BulkUploadDateFormat;
use App\Rules\ValidatePastDate;
use App\Services\Integrations\Tickets\TicketManagementService;
use Illuminate\Support\Facades\Auth;

class BulkLostStolenService extends BulkAbstract
{
    /**
     * Constructs a new instance of the class.
     *
     * @param TicketManagementService $ticketManagementService The TicketManagementService instance used for managing tickets.
     */
    public function __construct(protected TicketManagementService $ticketManagementService) {}


    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-lost-stolen-assets', compact('csvData', 'path', 'fileName'))->render();
    }


    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.lostStolenAssetData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return Config('tickets.bulk_lost_stolen');
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "stolen_lost";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "Stolen / Lost";
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'        => 'required|exists:assets,serial_no',
                'lost_date' => ['nullable', new BulkUploadDateFormat($data['lost_date'], config('date.formats.read_date_format'), $count, 'stolen/lost'), new ValidatePastDate($data['lost_date'], config('date.formats.read_date_format'), $count, 'stolen/lost')],
                'wipe_confirmation' => ['nullable', 'required_if:current_status_slug,retain_hold'],
                'retain_hold_wipe_confirmed_date' => ['nullable', 'required_if:current_status_slug,retain_hold', 'date_format:' . config('date.formats.read_date_format')],
                'retain_hold_note' => ['nullable', 'required_if:current_status_slug,retain_hold'],
            ],
            [
                'serial_no.required'    => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'      => 'Line no ' . $count . ' : The serial # does not exists',
                'wipe_confirmation.required_if' => 'Line no ' . $count . ' : The retain hold wipe confirmation is required when the current status is Retain Hold.',
                'retain_hold_wipe_confirmed_date.required_if' => 'Line no ' . $count . ' : The retain hold wipe confirmation date is required when the current status is Retain Hold.',
                'retain_hold_wipe_confirmed_date.date_format' => 'Line no ' . $count . ' : The retain hold wipe confirmation date must be in the format ' . config('date.formats.read_date_format') . '.',
                'retain_hold_note.required_if' => 'Line no ' . $count . ' : The retain hold notes is required when the current status is Retain Hold.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count     
     * @return array            
     */
    public function getData(array $inputData, int $count)
    {
        $assetData['asset_status_id'] = AssetStatus::where('slug', $this->getStatusSlug())->first()->id;
        $assetData['lost_date'] = $inputData['lost_date' . $count] ?? date("m/d/Y");
        $assetData['ticket_no'] = $inputData['ticket_no' . $count] ?? null;
        $assetData['wipe_confirmation'] = $inputData['wipe_confirmation' . $count] ?? null;
        $assetData['retain_hold_wipe_confirmed_date'] = $inputData['retain_hold_wipe_confirmed_date' . $count] ?? null;
        $assetData['retain_hold_note'] = $inputData['retain_hold_note' . $count] ?? null;

        return $assetData;
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk Stolen/Lost', Auth::user()->id, $filePath, $fileName);
        return true;
    }
}
