<?php

namespace App\Services\BulkUpload;

use Illuminate\Support\Facades\Validator;
use App\Models\AssetStatus;
use App\Services\Integrations\Tickets\TicketManagementService;
use Illuminate\Support\Facades\Auth;

class BulkLoansService extends BulkAbstract
{

    /**
     * Constructs a new instance of the BulkLoansService.
     *
     * @param TicketManagementService $ticketManagementService An instance of TicketManagementService used for managing ITSM tickets.
     */
    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * The function checks if a location in a CSV file is frozen for cycle count and returns any
     * errors.
     * 
     * @param csvDatas An array of data 
     * 
     * @return an array of errors.
     */
    public function checkNewLocationFreeze($csvDatas)
    {
        $errors = [];

        return ['errors' => $errors, 'csvData' => $csvDatas];
    }

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-loaned-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.loanedAssetData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        // return Config('tickets.bulk_loan');
        return "";
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "loaned";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "Loaned";
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'          => 'required|exists:assets,serial_no',
                'user_id'            => 'required|exists:users,id',
                'loaner_return_date' => ['required', 'date_format:' . config('date.formats.read_date_format')],
                // 'ticket_no'         => 'required'
            ],
            [
                'serial_no.required'          => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'            => 'Line no ' . $count . ' : The serial # does not exists',
                'user_id.required'            => 'Line no ' . $count . ' : The user is required.',
                'user_id.exists'              => 'Line no ' . $count . ' : The user does not exists',
                'loaner_return_date.required' => 'Line no ' . $count . ' : The expected return date is required.',
                'loaner_return_date.date_format' => 'Line no ' . $count . ' : The expected return date is not in ' . config('date.formats.read_date_format') . ' format.',
                'ticket_no.required'            => 'Line no ' . $count . ' : The ticket # required.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count     
     * @return array            
     */
    public function getData(array $inputData, int $count)
    {
        $assetData['asset_status_id'] = AssetStatus::where('slug', $this->getStatusSlug())->first()->id;
        $assetData['user_id'] = $inputData['user_id' . $count];
        $assetData['loaner_return_date'] = $inputData['loaner_return_date' . $count] ?? null;
        $assetData['ticket_no'] = $inputData['ticket_no' . $count] ?? null;

        return $assetData;
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk loan', Auth::user()->id, $filePath, $fileName);
        return true;
    }
}
