<?php

namespace App\Services\BulkUpload;

use App\Events\BulkUpdates;
use App\Models\Asset;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use App\Services\Integrations\Tickets\TicketManagementService;
use Illuminate\Support\Facades\Validator;

class BulkLinkService extends BulkAbstract
{
    public function __construct(protected TicketManagementService $ticketManagementService) {}

    public function getHeaderMap()
    {
        return Config('bulk-upload.linkData');
    }

    public function getDefaultTicket()
    {
        return Config('tickets.bulk_link');
    }

    /**
     * The function checks if a location in a CSV file is frozen for cycle count and returns any
     * errors.
     * 
     * @param csvDatas An array of data 
     * 
     * @return an array of errors.
     */
    public function checkNewLocationFreeze($csvDatas)
    {
        $errors = [];

        return ['errors' => $errors, 'csvData' => $csvDatas];
    }

    /**
     * The function checks if a location in a CSV file is frozen for cycle count and returns any
     * errors.
     * 
     * @param csvDatas An array of data 
     * 
     * @return an array of errors.
     */
    public function validateFromLocationFreeze($csvDatas)
    {
        $errors = [];

        return ['errors' => $errors, 'csvData' => $csvDatas];
    }

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-link-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Generate asset data from the csv array
     * @param  array $item
     * @param  int    $count
     * @return array
     */
    public function generateCSVData(array $item, int $count, $path = null, $fileName = null)
    {
        $assetData    = [];
        foreach ($this->getHeaderMap() as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);
            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "serial_no") {
                if (isset($item[$fileHeader]) && !empty($item[$fileHeader])) {
                    $assetData = $this->getAdditionalAssetData($assetData, $item[$fileHeader]);
                }
            }
            if ($dbField == 'parent_serial_no') {
                $assetData = $this->getAdditionalLinkData($assetData, $item[$fileHeader], 'parent');
            }
            if ($dbField == 'child_serial_no') {
                $assetData = $this->getAdditionalLinkData($assetData, $item[$fileHeader], 'child');
            }
        }

        $assetData['asset_count'] = $count++;

        if ($fileName && $path) {
            $assetData['filePath'] = $path;
            $assetData['fileName'] = $fileName;
        }

        session(['asset_count' => $assetData['asset_count']]);

        return compact('assetData');
    }

    /**
     * Get Additonal Asset Data.
     *
     * @return array
     */
    public function getAdditionalLinkData(array $assetData, string $item = '', string $type = 'parent')
    {
        $assetData[$type . '_asset_id'] = null;
        $assetData[$type . '_current_asset_status_id'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();
        if ($asset) {
            $assetData[$type . '_asset_id'] = $asset->id;
            $assetData[$type . '_current_asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'parent_serial_no' => 'required|exists:assets,serial_no',
                'child_serial_no' => 'required|exists:assets,serial_no',
            ],
            [
                'parent_serial_no.required' => 'Line no ' . $count . ' : The Parent Serial # is required.',
                'parent_serial_no.exists' => 'Line no ' . $count . ' : The Parent Serial # does not exists',
                'child_serial_no.required' => 'Line no ' . $count . ' : The Child Serial # is required.',
                'child_serial_no.exists' => 'Line no ' . $count . ' : The Child Serial # does not exists',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Check if the assets can be liked
     * @param  array $csvData
     * @return array
     */
    public function checkCanBeUpdatedToStatus($csvDatas)
    {
        return $this->canBeLinked($csvDatas);
    }

    /**
     * Checks if there are any duplicate serial numbers in the upload data.
     *
     * @param array $csvData The upload data.
     *
     * @return array An array containing the errors and the valid data.
     */
    public function checkRepeatUploadData($csvData)
    {
        $errors = [];
        return ['errors' => $errors, 'csvData' => $csvData];
    }

    /**
     * Checks if the given assets can be linked and gathers error information.
     *
     * @param array $csvDatas The CSV data containing asset information.
     *
     * @return array An array containing any error messages related to linking the assets.
     */
    public function canBeLinkedOld(array $csvDatas)
    {
        $errors = [];
        $rowCount = 1;

        foreach ($csvDatas as $csvData) {
            ++$rowCount;
            $data = $csvData['assetData'];
            if (isset($data['parent_asset_id'])) {
                if (Asset::where('serial_no', $data['parent_serial_no'])->parent()->count() < 1) {
                    $errors[] = 'Line no ' . ($rowCount) . ' : Attempting to link to an asset, which is not a parent.';
                }
                if (!Asset::find($data['parent_asset_id'])->canBeLinked()) {
                    $errors[] = 'Line no ' . ($rowCount) . ' : Parent asset can not link due to status restrictions.';
                }
            }
            if (isset($data['child_asset_id'])) {
                if (Asset::where('serial_no', $data['child_serial_no'])->child()->count() < 1) {
                    $errors[] = 'Line no ' . ($rowCount) . ' : Attempting to link an asset, which is not a child.';
                }
                if (!Asset::find($data['child_asset_id'])->canBeLinked()) {
                    $errors[] = 'Line no ' . ($rowCount) . ' : Child asset can not link due to status restrictions.';
                }
            }
        }

        return $errors;
    }

    /**
     * Checks if the given assets can be linked and gathers error information.
     *
     * @param array $csvDatas The CSV data containing asset information.
     *
     * @return array An array containing any error messages related to linking the assets and the valid data that can be linked.
     */
    private function canBeLinked(array $csvDatas)
    {
        $errors = [];
        $validData = [];

        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $rowCount = $data['asset_count'] + 1;

            // Validate Parent Asset
            if (isset($data['parent_serial_no'])) {
                $parentAsset = Asset::where('serial_no', $data['parent_serial_no'])->parent()->first();

                if (!$parentAsset) {
                    $errors[$rowCount] = 'Line no ' . $rowCount . ' : Attempting to link to an asset that is not a parent or does not exist.';
                    continue;
                }

                if (!$parentAsset->canBeLinked()) {
                    $errors[$rowCount] = 'Line no ' . $rowCount . ' : Parent asset cannot be linked due to status restrictions.';
                    continue;
                }
            }

            // Validate Child Asset
            if (isset($data['child_serial_no'])) {
                $childAsset = Asset::where('serial_no', $data['child_serial_no'])->child()->first();

                if (!$childAsset) {
                    $errors[$rowCount] = 'Line no ' . $rowCount . ' : Attempting to link an asset that is not a child or does not exist.';
                    continue;
                }

                if (!$childAsset->canBeLinked()) {
                    $errors[$rowCount] = 'Line no ' . $rowCount . ' : Child asset cannot be linked due to status restrictions.';
                    continue;
                }
            }

            // If all validations pass, add to valid data
            $validData[] = $csvData;
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }



    /**
     * Save Data from the request
     * @param  array  $inputData
     * @param  int    $i
     * @return boolean
     */
    public function saveData(array $inputData, int $i)
    {
        $asset = Asset::find($inputData['child_asset_id' . $i]);
        $parentAsset = Asset::find($inputData['parent_asset_id' . $i]);

        if ($asset === null || $parentAsset === null) {
            return false;
        }

        if (!$asset->canBeLinked() || !$parentAsset->canBeLinked()) {
            return false;
        }

        $this->updateAsset($asset, $parentAsset);
        $this->addAssetHistory($asset, $parentAsset, $inputData, $i);

        return true;
    }

    public function addAssetHistory(Asset $asset, Asset $parentAsset, array $inputData, int $i)
    {
        $description = __('history.Linked', [
            'parentAssetId' => $parentAsset->id,
            'childAssetId' => $asset->id,
            'parentAssetname' => $parentAsset->serial_no,
            'childAssetname' => $asset->serial_no,
            'commenttext' => '',
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $inputData['ticket_no' . $i] ?? null,
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'linked',
            'comments' => '',
            'old_parent_asset_id' => $asset->parent_asset_id,
            'new_parent_asset_id' => $parentAsset->id,
            'old_location_id' => $asset->location_id,
            'new_location_id' => $parentAsset->location_id,
            'old_user_id' => $asset->user_id,
            'new_user_id' => $parentAsset->user_id,
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $parentAsset->asset_status_id,
            'old_value' => $parentAsset->serial_no,
            'new_value' => $asset->serial_no,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
    }

    /**
     * Update Asset
     * @param  Asset  $asset
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function updateAsset(Asset $asset, Asset $parentAsset)
    {
        $carrierId = ($parentAsset->carrier_id != null) ? $parentAsset->carrier_id : $asset->carrier_id;
        $asset->update([
            'parent_asset_id' => $parentAsset->id,
            'user_id' => $parentAsset->user_id,
            'location_id' => $parentAsset->location_id,
            'asset_status_id' => $parentAsset->asset_status_id,
            'carrier_id' => $carrierId,
            'linked_date' => Carbon::now(),
        ]);
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "";
    }


    /**
     * Get Headers of CSV
     * @return array
     */
    public function getData(array $inputData, int $count)
    {
        return "";
    }


    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk link', Auth::user()->id, $filePath, $fileName);
        return true;
    }
}
