<?php

namespace App\Services\BulkUpload;

use Illuminate\Support\Facades\Validator;
use App\Models\AssetStatus;
use Illuminate\Support\Facades\Auth;
use App\Services\Integrations\Tickets\TicketManagementService;

class BulkInstallService extends BulkAbstract
{
    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-install-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    public function csvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no' => 'required|exists:assets,serial_no',
                'location_id' => 'required|exists:locations,id',
            ],
            [
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'    => 'Line no ' . $count . ' : The serial # does not exist.',
                'location_id.required' => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'    => 'Line no ' . $count . ' : The location does not exist.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.installAssetData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return Config('tickets.bulk_install');
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "installed";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "Installed";
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count     
     * @return array            
     */
    public function getData(array $inputData, int $count)
    {
        $assetData['asset_status_id'] = AssetStatus::where('slug', $this->getStatusSlug())->first()->id;
        $assetData['location_id']  = $inputData['location_id' . $count];
        $assetData['ticket_no'] = $inputData['ticket_no' . $count] ?? null;

        return $assetData;
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk install', Auth::user()->id, $filePath, $fileName);
        return true;
    }
}
