<?php

namespace App\Services\BulkUpload;

use App\Http\Traits\AssetStatusTrait;
use App\Models\Asset;
use App\Rules\BulkUploadDateFormat;
use Illuminate\Support\Facades\Auth;
use App\Services\Integrations\Tickets\TicketManagementService;
use Illuminate\Support\Facades\Validator;

class BulkHoldService extends BulkAbstract
{
    use AssetStatusTrait;

    /**
     * Construct a new instance of the class.
     *
     * @param TicketManagementService $ticketManagementService The TicketManagementService instance used for managing tickets.
     */
    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-hold-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.holdAssetData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        // return Config('tickets.bulk_hold');
        return "";
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "";
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'          => 'required|exists:assets,serial_no',
                'location_id'        => 'required|exists:locations,id',
                'user_id'            => 'nullable|exists:users,id',
                'status_slug'        => 'required|in:legal_hold,retain_hold',
                'loaner_retention_date' => ['nullable', new BulkUploadDateFormat($data['loaner_retention_date'], config('date.formats.read_date_format'), $count, 'Retain Hold Release')],
                // 'ticket_no'         => 'required'
            ],
            [
                'serial_no.required'          => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'            => 'Line no ' . $count . ' : The serial # does not exists',
                'location_id.required'        => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'          => 'Line no ' . $count . ' : The location does not exists',
                'user_id.required'            => 'Line no ' . $count . ' : The user is required.',
                'user_id.exists'              => 'Line no ' . $count . ' : The user does not exists',
                'asset_status_id.required'    => 'Line no ' . $count . ' : The status is required.',
                'ticket_no.required'            => 'Line no ' . $count . ' : The ticket # required.',
                'status_slug.in'              => 'Line no ' . $count . ' : The status is should be Legal Hold or Retain Hold.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count     
     * @return array            
     */
    public function getData(array $inputData, int $count)
    {
        $assetData['asset_status_id']         = $inputData['asset_status_id' . $count];
        $assetData['user_id']                 = $inputData['user_id' . $count];
        $assetData['location_id']              = $inputData['location_id' . $count];
        if ($inputData['status_name' . $count] === "Retain Hold") {
            $assetData['loaner_retention_date'] = $inputData['loaner_retention_date' . $count] ?? null;
        }
        $assetData['ticket_no']             = $inputData['ticket_no' . $count] ?? null;

        return $assetData;
    }

    /**
     * Check if the asset can be updated to the status
     * @param  array $csvData
     * @return array
     */
    public function checkCanBeUpdatedToStatus($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        $validData = [];
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];
            if ($data['asset_status_id']) {
                $canUpdatedStatuses = $this->searchFromStatus($this->getStatus($data['asset_status_id'])->slug);
                $currentStatusSlug = $this->getStatus($data['current_asset_status_id'])->slug;

                if (!in_array($currentStatusSlug, $canUpdatedStatuses)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The asset with serial # : ' . $data['serial_no'] . ' cannot be updated to ' . $this->getStatus($data['asset_status_id'])->name . '.';
                    continue;
                }

                $validData[] = $csvData;
            }
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }


    /**
     * Get Additonal Asset Data
     * @param  array  $assetData
     * @param  string $item
     * @return array
     */
    public function getAdditionalAssetData(array $assetData, string $item = "")
    {
        $assetData['asset_id'] = null;
        $assetData['current_asset_status_id'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();
        if ($asset) {
            $assetData['asset_id'] = $asset->id;
            $assetData['current_asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    /**
     * Add Additional Data to show in the uploaded view
     * @param array $assetData
     * @return array
     */
    public function addAdditionalData(array $assetData)
    {

        $assetData['current_status_slug'] = isset($assetData['current_asset_status_id']) ? $this->getStatus($assetData['current_asset_status_id'])->slug : '';
        $assetData['current_status_name'] = isset($assetData['current_asset_status_id']) ? $this->getStatus($assetData['current_asset_status_id'])->name : '';
        $assetData['status_slug'] = isset($assetData['asset_status_id']) ? $this->getStatus($assetData['asset_status_id'])->slug : '';
        $assetData['status_name'] = isset($assetData['asset_status_id']) ? $this->getStatus($assetData['asset_status_id'])->name : '';

        return $assetData;
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk hold', Auth::user()->id, $filePath, $fileName);
        return true;
    }
}
