<?php

namespace App\Services\BulkUpload;

use App\Events\BulkUpdates;
use App\Models\AssetStatus;
use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;
use App\Services\Integrations\Tickets\TicketManagementService;
use Carbon\Carbon;
use Facades\App\Repositories\BulkUpload;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class BulkHardwareStandardsService extends BulkAbstract
{
    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * Return the view for displaying errors or data
     * @param  string $path
     * @return json
     */
    public function getViewData(string $path, $fileName)
    {
        $data = $this->importAssetData($path);

        if ($view = $this->handleImportErrors($data)) {
            return response()->json($view);
        }

        $csvData = $data['csvData'];

        if ($view = $this->handleCsvColumnErrors($path, $csvData)) {
            return response()->json($view);
        }

        if ($csvData) {

            $validatedData = $this->validateFields($csvData);

            $errors = $validatedData['errors'];
            $csvData = $validatedData['csvData'];

            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            $view['data'] = $this->renderCsvData($csvData, $path, $fileName);
            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.uploaded-hardwares-standards', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Return data from csv
     * @param  string $path
     * @return array
     */
    public function importAssetData(string $path, $fileName = null)
    {
        $count = 1;
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];

        if (!empty($items)) {
            foreach ($items as $key => $item) {
                if (!count($item)) {
                    continue; //skip empty columns
                }
                $csvData['csvData'][] = $this->generateCSVData($item, $count++, $path, $fileName);
            }
        }
        return $csvData;
    }

    /**
     * Generate asset data from the csv array
     * @param array $item
     * @param  int   $count
     * @return array
     */
    public function generateCSVData($item, int $count, $path = null, $fileName = null)
    {
        $assetData    = [];
        $assetData = array_combine(
            array_map(fn($key) => str_slug($key, '_'), array_keys($item)),
            $item
        );
        foreach ($this->getHeaderMap() as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);
            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? trim($item[$fileHeader]) : null;
        }
        if ($fileName && $path) {
            $assetData['filePath'] = $path;
            $assetData['fileName'] = $fileName;
        }

        $assetData['asset_count'] = $count++;
        session(['asset_count' => $assetData['asset_count']]);

        return compact('assetData');
    }

    /**
     * Get Related Values for ids.
     * @param  string $item
     * @param  string $dbField
     * @param  string $fileHeader
     *
     * @return mixed
     */
    public function getRelationalValues($item,  $dbField,  $fileHeader)
    {
        if ($dbField == "asset_type_id") {
            return BulkUpload::getAssetTypeId($item, $dbField, $fileHeader);
        }

        if ($dbField == "manufacturer_id") {
            return BulkUpload::getManufacturerId($item, $dbField, $fileHeader);
        }

        if ($dbField == "make_and_model_id") {
            return BulkUpload::getAttributeMakeModelId($item, $dbField, $fileHeader);
        }

        return false;
    }

    public function csvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                'asset_type_id' => ['required'],
                'manufacturer_id' => ['required'],
                'make_and_model_id' => [Rule::requiredIf(function () use ($data) {
                    return !empty($data['current_hardware_standard']);
                })],
                'new_hardware_standard' => ['required'],
                'status' => ['nullable', Rule::in(['Active', 'Inactive'])]
            ],
            [
                'asset_type_id.required' => 'Line no ' . $count . ' : The asset type is required.',
                'manufacturer_id.required'    => 'Line no ' . $count . ' The manufacturer is required.',
                'make_and_model_id.required'    => 'Line no ' . $count . ' The hardware standard does not exist.',
                'new_hardware_standard.required'    => 'Line no ' . $count . ' : The new hardware standard is required.',
                'status.in' => 'Line no ' . $count . ' : The status must be either "Active" or "Inactive".',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.hardwareStandards');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return "";
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "";
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count     
     * @return array            
     */
    public function getData(array $inputData, int $count)
    {
        return [];
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk hardware standards update', Auth::user()->id, $filePath, $fileName);
        return true;
    }

    /**
     * Save Data from the request
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveData(array $inputData, int $i)
    {
        try {
            if ($inputData['make_and_model_id' . $i]) {
                $makeAndModel = MakeAndModel::with(['manufacturer', 'assetType'])->where('id', $inputData['make_and_model_id' . $i])->first();
                $hardwareData = [
                    'asset_type_id' => $inputData['asset_type_id' . $i] ?: $makeAndModel->asset_type_id,
                    'manufacturer_id' => $inputData['manufacturer_id' . $i] ?: $makeAndModel->manufacturer_id,
                    'name' => $inputData['new_hardware_standard' . $i],
                    'slug' => str_slug($inputData['new_hardware_standard' . $i], '_'),
                    'status' => $inputData['status' . $i] ? ($inputData['status' . $i] == 'Active' ? 1 : 0) : $makeAndModel->status,
                ];
                $this->createUpdateHistory($inputData, $makeAndModel, $i);
                MakeAndModel::where('id', $inputData['make_and_model_id' . $i])->update($hardwareData);
            } else {

                $hardwareData = [
                    'asset_type_id' => $inputData['asset_type_id' . $i],
                    'manufacturer_id' => $inputData['manufacturer_id' . $i],
                    'name' => $inputData['new_hardware_standard' . $i],
                    'slug' => str_slug($inputData['new_hardware_standard' . $i], '_'),
                    'status' => $inputData['status' . $i] == 'Active' ? 1 : 0,
                ];

                $this->createHistory($inputData, $i);
                $makeAndModelId = MakeAndModel::UpdateOrCreate($hardwareData)->id;
                $techSpecData = [
                    'make_and_model_id' => $makeAndModelId,
                    'details' => $inputData['technical_specs' . $i],
                    'original_value' => 0,
                ];

                TechnicalSpecs::create($techSpecData);
            }

            return true;
        } catch (\Exception) {
            return false;
        }
    }

    /**
     * Creates the hardware update history
     * @param mixed $inputData
     * @param mixed $existingdata
     * @param mixed $i
     * 
     * @return [type]
     */
    private function createUpdateHistory($inputData, $existingdata, $i)
    {
        $description = '';

        if ($inputData['manufacturer' . $i] != optional($existingdata->manufacturer)->name) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Manufacturer',
                'oldValue'    => optional($existingdata->manufacturer)->name,
                'newValue'  => $inputData['manufacturer' . $i],
            ]);
        }
        if (($inputData['current_hardware_standard' . $i] && $inputData['new_hardware_standard' . $i]) && ($inputData['new_hardware_standard' . $i] != $inputData['current_hardware_standard' . $i])) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Make and Model',
                'oldValue'    => $inputData['current_hardware_standard' . $i],
                'newValue'  => $inputData['new_hardware_standard' . $i],
            ]);
        }
        if ($inputData['asset_type' . $i] != optional($existingdata->assetType)->name) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Asset Type',
                'oldValue'    => optional($existingdata->assetType)->name,
                'newValue'  => $inputData['asset_type' . $i],
            ]);
        }
        if (($inputData['status' . $i] && $existingdata->status) && ($inputData['status' . $i] != $existingdata->status)) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Status',
                'oldValue'    => $existingdata->status == 0 ? 'Inactive' : 'Active',
                'newValue'  => $inputData['status' . $i],
            ]);
        }

        $description = __('history.HardwareStandardUpdated', [
            'description' => $description
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'hardware_standard_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
        return true;
    }

    /**
     * creates new hardware created history
     * @param mixed $inputData
     * @param int $i
     * 
     * @return [type]
     */
    private function createHistory($inputData, $i)
    {
        $description = __('history.HardwareStandardCreated', [
            'assetType' => $inputData['asset_type' . $i],
            'manufacturer' => $inputData['manufacturer' . $i],
            'make_model' => $inputData['new_hardware_standard' . $i]
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'hardware_standard_created',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
        return true;
    }
}
