<?php

namespace App\Services\BulkUpload;

use App\Http\Traits\AssetStatusTrait;
use App\Models\AssetStatus;
use App\Models\Asset;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Services\Integrations\Tickets\TicketManagementService;

class BulkDamageService extends BulkAbstract
{
    use AssetStatusTrait;

    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * Renders the CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-damaged-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.damagedAssetData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        // return Config('tickets.bulk_damage');
        return "";
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "damaged";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "";
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'          => 'required|exists:assets,serial_no',
                'location_id'        => 'required|exists:locations,id',
                'asset_status_id'    => 'required',
                // 'ticket_no'          => 'required',
                'wipe_confirmation' => ['nullable', 'required_if:current_status_slug,retain_hold'],
                'retain_hold_wipe_confirmed_date' => ['nullable', 'required_if:current_status_slug,retain_hold', 'date_format:' . config('date.formats.read_date_format')],
                'retain_hold_note' => ['nullable', 'required_if:current_status_slug,retain_hold'],
            ],
            [
                'serial_no.required'          => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'            => 'Line no ' . $count . ' : The serial # does not exists',
                'location_id.required'        => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'          => 'Line no ' . $count . ' : The location does not exists',
                'asset_status_id.required'    => 'Line no ' . $count . ' : The status is required.',
                'ticket_no.required'            => 'Line no ' . $count . ' : The ticket # required.',
                'wipe_confirmation.required_if' => 'Line no ' . $count . ' : The retain hold wipe confirmation is required when the current status is Retain Hold.',
                'retain_hold_wipe_confirmed_date.required_if' => 'Line no ' . $count . ' : The retain hold wipe confirmation date is required when the current status is Retain Hold.',
                'retain_hold_wipe_confirmed_date.date_format' => 'Line no ' . $count . ' : The retain hold wipe confirmation date must be in the format ' . config('date.formats.read_date_format') . '.',
                'retain_hold_note.required_if' => 'Line no ' . $count . ' : The retain hold notes is required when the current status is Retain Hold.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count     
     * @return array            
     */
    public function getData(array $inputData, int $count)
    {
        $assetData['asset_status_id']     = $inputData['asset_status_id' . $count];
        $assetData['location_id']          = $inputData['location_id' . $count];
        $assetData['ticket_no']         = $inputData['ticket_no' . $count] ?? null;

        return $assetData;
    }


    /**
     * Get Additonal Asset Data
     * @param  array  $assetData
     * @param  string $item
     * @return array
     */
    public function getAdditionalAssetData(array $assetData, string $item = "")
    {
        $assetData['asset_id'] = null;
        $assetData['current_asset_status_id'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();
        if ($asset) {
            $assetData['asset_id'] = $asset->id;
            $assetData['current_asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    /**
     * Add Additional Data to show in the uploaded view
     * @param array $assetData
     * @return array
     */
    public function addAdditionalData(array $assetData)
    {

        $assetData['current_status_slug'] = isset($assetData['current_asset_status_id']) ? $this->getStatus($assetData['current_asset_status_id'])->slug : '';
        $assetData['current_status_name'] = isset($assetData['current_asset_status_id']) ? $this->getStatus($assetData['current_asset_status_id'])->name : '';
        $assetData['status_slug'] = isset($assetData['asset_status_id']) ? $this->getStatus($assetData['asset_status_id'])->slug : '';
        $assetData['status_name'] = isset($assetData['asset_status_id']) ? $this->getStatus($assetData['asset_status_id'])->name : '';

        return $assetData;
    }

    /**
     * Check if the asset can be updated to the status
     * @param  array $csvData
     * @return array
     */
    public function checkCanBeUpdatedToStatus($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        $validData = [];

        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $rowCount = $data['asset_count'] + 1;
            if (isset($data['current_asset_status_id']) && $data['current_asset_status_id']) {

                $canUpdatedStatuses = $this->searchFromStatus($this->getStatusSlug());
                $currentStatusSlug = AssetStatus::find($data['current_asset_status_id'])->slug;

                if (!in_array($currentStatusSlug, $canUpdatedStatuses)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The serial # : ' . $data['serial_no'] . ' cannot be updated to ' . $this->getStatus($data['asset_status_id'])->name . '.';
                    continue;
                }

                $validData[] = $csvData;
            }
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }

    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk damage', Auth::user()->id, $filePath, $fileName);

        return true;
    }
}
