<?php

namespace App\Services\BulkUpload;

use Illuminate\Support\Facades\Validator;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Http\Traits\BulkUploadTrait;
use App\Models\AssetType;
use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;
use App\Services\Asset\AssetStatusHistoryService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class BulkCertificatesUpload
{

    use BulkUploadTrait;

    protected $updated;
    protected $notFound;
    protected $duplicate;
    protected $certificatePath;

    /**
     * Constructor for BulkCertificatesUpload.
     * 
     * @param AssetStatusHistoryService $assetStatusHistoryService Instance of AssetStatusHistoryService.
     */
    public function __construct(
        protected AssetStatusHistoryService $assetStatusHistoryService
    ) {

        $this->updated = 0;
        $this->notFound = '';
        $this->duplicate = '';
        $this->certificatePath = '';
    }

    /**
     * Returns a JSON response containing the view data for a given path and file name.
     * This method is used for bulk uploading certificates.
     *
     * @param string $path The path of the file.
     * @param string $certificatePath The path to the certificate file.
     * @return \Illuminate\Http\JsonResponse The JSON response containing the view data.
     */
    public function getViewData($path, $certificatePath)
    {
        $view = [];
        $data = $this->importAssetData($path);

        if ($view = $this->handleImportErrors($data)) {
            return response()->json($view);
        }

        $csvData = $data['csvData'];

        if ($view = $this->handleCsvColumnErrors($path, $csvData)) {
            return response()->json($view);
        }

        if ($csvData) {

            $this->certificatePath = str_replace("public/", "", $certificatePath);

            return $this->saveData($csvData);
        }

        return response()->json($view);
    }

    /**
     * Import asset data from a given file path.
     *
     * @param string $path The path to the file containing asset data.
     *
     * @return array An array containing the CSV data and any errors encountered.
     */
    public function importAssetData($path)
    {
        $count = 1;
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {
            foreach ($items as $item) {
                if (!count($item)) continue; //skip empty columns
                $csvData['csvData'][] = $this->generateCSVData($item, $count++);
            }
        }

        return $csvData;
    }

    /**
     * Generate asset data from the csv array
     * @param array $item A row of data from the CSV file.
     * @param int $count The current row count.
     * @return array An array containing the asset data for the given row.
     */
    public function generateCSVData($item, $count)
    {
        if (empty($item)) {
            return false;
        }

        $headerMap = $this->getHeaderMap();

        $assetData = [];
        foreach ($headerMap as $dbField => $fileHeader) {
            $assetData[$dbField] = $this->getFieldValue($item, $fileHeader);

            if ($dbField === "serial_no" && !empty($item[$fileHeader])) {
                $assetDetails = $this->getAssetDetailsBySerial($item[$fileHeader]);
                $assetData = array_merge($assetData, $assetDetails);
            }
        }

        $assetData['asset_count'] = $count++;
        session(['asset_count' => $assetData['asset_count']]);

        return compact('assetData');
    }

    /**
     * Sub-function to retrieve a value from the item array based on the header key.
     */
    private function getFieldValue($item, $fileHeader)
    {
        return !empty($item[$fileHeader]) ? $item[$fileHeader] : null;
    }

    /**
     * Sub-function to find asset details by serial number.
     */
    private function getAssetDetailsBySerial($serialNo)
    {
        $serialNo = trim($serialNo);
        $asset = Asset::where('serial_no', $serialNo)->first();

        return [
            'asset_id' => $asset->id ?? null,
            'asset_status_id' => $asset->asset_status_id ?? null,
            'serial_no' => $serialNo,
        ];
    }

    /**
     * Validate each row of the CSV data for bulk uploading certificates.
     *
     * @param array $data The CSV data to validate.
     * @param int $count The row count for tracking purposes.
     *
     * @return array The validation errors, if any.
     */
    public function csvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no' => 'required|exists:assets,serial_no',
            ],
            [
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'    => 'Line no ' . $count . ' : The serial # does not exist.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Returns the header map for the CSV file used for bulk uploading certificates.
     *
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.certificatesUploadData');
    }

    /**
     * Saves the data from the csv to the database. This method is used for bulk uploading certificates.
     *
     * @param array $csvData The csv data to be saved.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the outcome of the bulk upload.
     */
    public function saveData($csvData)
    {
        $updatedAssets = [];

        // Cache the status ID with a unique key and a specified expiration time
        $statusId = Cache::remember('end_of_life_data_wiped_status_id', 60 * 60, function () {
            return AssetStatus::getStatusId('end_of_life_data_wiped')->first()->id;
        });


        foreach ($csvData as $csvData) {
            if (!in_array($csvData['assetData']['serial_no'], $updatedAssets)) {
                if ($csvData['assetData']['asset_id'] == null) {
                    // $csvData = $this->createAsset($csvData);
                }

                try {
                    $asset = Asset::with('assetStatus')->findOrFail($csvData['assetData']['asset_id']);

                    $this->updateAssetData($asset, $statusId);
                    $this->updated++;

                    array_push($updatedAssets, $csvData['assetData']['serial_no']);
                } catch (Exception $e) {
                    $this->notFound .= '#' . $csvData['assetData']['serial_no'] . ', ';
                }
            } else {
                $this->duplicate .= '#' . $csvData['assetData']['serial_no'] . ', ';
            }
        }

        return $this->createResponseMessage();
    }

    /**
     * Create dummy asset if Serial not exist.
     *
     * @param
     *
     * @return array
     */
    public function createAsset($csvData)
    {
        $statusId = AssetStatus::getStatusId('end_of_life_data_wiped')->first()->id;
        $assetType = AssetType::whereSlug('eol_asset_codd')->first()->id;
        $makeAndModel = MakeAndModel::whereSlug('eol_asset_codd')->first()->id;
        $technicalSpecs = TechnicalSpecs::whereDetails('EOL Asset - CODD')->first()->id;

        $asset = [
            'serial_no' => trim($csvData['assetData']['serial_no']),
            'asset_tag' => trim($csvData['assetData']['serial_no']),
            'asset_status_id' => $statusId,
            'asset_type_id' => $assetType,
            'make_and_model_id' => $makeAndModel,
            'technical_spec_id' => $technicalSpecs,
        ];

        $asset = Asset::Create($asset);

        $csvData['assetData']['asset_id'] = $asset->id;
        $csvData['assetData']['asset_status_id'] = $asset->statusId;

        return $csvData;
    }

    /**
     * Update asset details
     *
     * @param App\Models\Asset $asset
     * @param int App\Models\AssetStatus id
     */
    public function updateAssetData($asset, $statusId)
    {
        $this->assetStatusHistoryService->createEolCertificateUploadHistory($asset, $statusId);

        $asset->update([
            'certificate_added_date' => Carbon::now()->format('m/d/Y'),
            'ewaste_certificate' => $this->certificatePath,
            'location_id' => null,
            'asset_status_id' => $statusId,
            'loaner_return_date' => null,
            'loaner_retention_date' => null,
            'user_id' => null,
            'lost_date' => null,
            'date_deployed' => null,
        ]);
    }

    /**
     *Making json response with message
     */
    public function createResponseMessage()
    {
        Log::channel('single')->info("EOL Bulk Certificate Upload - Total certificates added - " . $this->updated);
        $response['success_message'] = $this->updated . ' Assets Updated';

        if ($this->notFound) {
            $response['error_message']   = 'Assets with serial # ' . rtrim($this->notFound, ', ') . ' do not exist';
        }

        if ($this->duplicate) {
            $response['error_message']   .= 'Serial # ' . rtrim($this->duplicate, ', ') . ' is updated but has duplicate entries in the uploaded file.';
        }

        return $response;
    }

    /**
     * Create new folder, if not exists
     */
    public function createFolder()
    {
        $forlderName = date('Y_m_d');
        $path        = '/public/uploads/eol_certificates/' . $forlderName;
        Storage::makeDirectory($path);

        return $path;
    }
}
