<?php

namespace App\Services\BulkUpload;

use App\Http\Traits\AssetAttributeUpdateTrait;
use App\Events\BulkUpdates;
use App\Models\Asset;
use App\Rules\ValidateMakeModelAssetTypeMatch;
use App\Rules\ValidateTechSpecsMakeModelMatch;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use App\Services\Integrations\Tickets\TicketManagementService;
use Illuminate\Support\Facades\Auth;

class BulkAssetAttributeService extends BulkAbstract
{
    use AssetAttributeUpdateTrait;

    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * Return the view for displaying errors or data
     * @param  string $path
     * @return json
     */
    public function getViewData(string $path, $fileName)
    {
        $data = $this->importAssetData($path, $fileName);
        if ($view = $this->handleImportErrors($data)) {
            return response()->json($view);
        }

        $csvData = $data['csvData'];

        if ($view = $this->handleCsvColumnErrors($path, $csvData)) {
            return response()->json($view);
        }

        if ($csvData) {
            $validatedData = $this->validateFields($csvData);

            $errors = $validatedData['errors'];
            $csvData = $validatedData['csvData'];
            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            $duplicationValidatedData = $this->checkRepeatUploadData($csvData);
            $csvData = $duplicationValidatedData['csvData'];
            $duplicateErrors = $duplicationValidatedData['errors'];
            $view['duplicate_errors'] = view('assets.partials.upload-duplicate-errors', compact('duplicateErrors'))->render();

            $view['data'] = $this->renderCsvData($csvData, $path, $fileName);

            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }


    /**
     * Renders the given CSV data into a view and returns the rendered content.
     *
     * @param array $csvData The CSV data to be rendered.
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return string The rendered content of the view.
     */
    public function renderCsvData($csvData, $path, $fileName)
    {
        return view('settings.partials.upload-attribute-update-assets', compact('csvData', 'path', 'fileName'))->render();
    }

    /**
     * Get Headers of CSV
     * @return array
     */
    public function getHeaderMap()
    {
        return Config('bulk-upload.attributeUpdateData');
    }

    /**
     * Get Default Ticket #
     * @return String
     */
    public function getDefaultTicket()
    {
        return '';
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getStatusSlug()
    {
        return "";
    }

    /**
     * Get Status Name
     * @return string
     */
    public function getStatusName()
    {
        return "";
    }

    /**
     * Get Status Slug
     * @return string
     */
    public function getData(array $inputData, int $count)
    {
        return "";
    }

    /**
     * Validate each row
     * @param  array $data
     * @param  int $count
     * @return object
     */
    public function csvValidator(array $data, int $count)
    {
        $validator = Validator::make(
            $data,
            [
                'asset_id'             => ['required', 'exists:assets,id'],
                'serial_no'            => ['required', 'exists:assets,serial_no'],
                'asset_tag'            => ['nullable', 'unique:assets,asset_tag'],
                'asset_type_id'        => ['nullable', 'exists:asset_types,id'],
                'make_and_model_id'    => ['nullable', 'exists:make_and_models,id', 'required_with:asset_type_id', new ValidateMakeModelAssetTypeMatch($data['asset_type_id'], $data['asset_id'], $count)],
                'technical_spec_id'    => ['nullable', 'exists:technical_specs,id', 'required_with:make_and_model_id', new ValidateTechSpecsMakeModelMatch($data['make_and_model_id'], $data['asset_id'], $count)],
                'carrier_id'           => ['nullable', 'exists:carriers,id'],
                'legalhold'            => ['nullable', 'exists:users,legalhold,0'],
                'created_at'           => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'warranty_end_date'    => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                // 'ticket_no'            => ['required'],
            ],
            [
                'serial_no.required'          => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'            => 'Line no ' . $count . ' : The serial # does not exists',
                'asset_tag.distinct' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'asset_tag.unique' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'make_and_model_id.exists'    => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.exists'    => 'Line no ' . $count . ' : The technical specs does not exist.',
                'asset_type_id.exists'    => 'Line no ' . $count . ' : The asset type does not exist.',
                'carrier_id.exists'    => 'Line no ' . $count . ' : The carrier does not exist.',
                'legalhold'  => 'Line no ' . $count . ' : Assets on legal hold employees can not be edited.',
                'created_at.date_format'    => 'Line no ' . $count . ' : The created date format is incorrect.Use ' . config('date.formats.read_date_format') . ' format',
                'warranty_end_date.date_format'    => 'Line no ' . $count . ' : The warranty end date format is incorrect.Use Use ' . config('date.formats.read_date_format') . ' format',
                'ticket_no.required'          => 'Line no ' . $count . ' : The ticket # is required.',
                'make_and_model_id.required_with' => 'Line no ' . $count . ' : The hardware standard is required to change asset type.',
                'technical_spec_id.required_with' => 'Line no ' . $count . ' : The technical specs is required to change hardware standard.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Save Data from the request
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveData(array $inputData, int $i)
    {
        $asset = Asset::find($inputData['asset_id']);
        if ($asset === null) {
            return false;
        }

        $data  = $this->getAttributeData($inputData, $asset, $i);

        if ($data['asset'] && !empty($data['asset'])) {
            $this->saveAssetHistory($asset, $data);
            $asset->update($data['asset']);
        }

        return true;
    }

    /**
     * Saves the history of the given asset with the provided data.
     *
     * @param Asset $asset The asset for which the history is being saved.
     * @param array $data  The data related to the asset changes, including history details.
     *
     * @return void|null Returns null if no history is provided.
     */
    public function saveAssetHistory(Asset $asset, array $data)
    {
        $history = rtrim(trim($data['history']), ',');
        if (!$history) {
            return null;
        }
        $description = "Updated attributes of asset <a href='" . route('assets.show', $asset->id) . "'>$asset->asset_tag</a> " . $history;

        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $data['asset']['ticket_no'] ?? null,
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'assetAttributes_changed',
            'comments' => '',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        $assetHistory = $this->getAdditionalHistoryData($assetHistory, $data, $asset);
        event(new BulkUpdates($assetHistory));
    }


    /**
     * After bulk upload attach the uploaded file to the ticket as comment
     * @param mixed $ticketid
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function attachFiletoTicket($ticketid, $filePath, $fileName)
    {
        if (!$ticketid || !$filePath || !$fileName) {
            return false;
        }
        $this->ticketManagementService->attachActionHistorytoTicket($ticketid, 'Bulk asset attribute update', Auth::user()->id, $filePath, $fileName);

        return true;
    }
}
