<?php

namespace App\Services\BulkUpload;

use App\Models\Asset;
use App\Models\AssetStatus;
use Illuminate\Support\Facades\Auth;
use Facades\App\Services\AssetHistory as AssetHistoryService;
use Facades\App\Services\Asset\AssetStatusService;
use App\Http\Traits\BulkUploadTrait;
use Carbon\Carbon;
use App\Events\BulkUpdates;
use App\Http\Traits\AssetStatusTrait;
use App\Models\Location;
use Exception;
use Facades\App\Services\Asset\UpdateStatusService;

abstract class BulkAbstract
{
    use BulkUploadTrait, AssetStatusTrait;

    abstract protected function getHeaderMap();
    abstract protected function getDefaultTicket();
    abstract protected function getStatusSlug();
    abstract protected function getStatusName();
    abstract protected function getData(array $inputData, int $count);
    abstract protected function attachFiletoTicket($ticketid, $filePath, $fileName);
    abstract protected function renderCsvData($csvData, $path, $fileName);

    /**
     * Return data from csv
     * @param  string $path
     * @param  string $fileName
     * @return array
     */
    public function importAssetData(string $path, $fileName = null)
    {
        $count = session('asset_count') ?? 1;
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];

        if (!empty($items)) {
            try {
                foreach ($items as $item) {
                    if (!count($item)) {
                        continue; //skip empty columns
                    }
                    $csvData['csvData'][] = $this->generateCSVData($item, $count++, $path, $fileName);
                }
            } catch (Exception) {
                $csvData['error'] = "There was an error while importing the file. Please try with the correct template.";
            }
        }

        return $csvData;
    }

    /**
     * Returns a JSON response containing the view data for a given path and file name.
     *
     * @param string $path The path of the file.
     * @param string $fileName The name of the file.
     * @return \Illuminate\Http\JsonResponse The JSON response containing the view data.
     */
    public function getViewData(string $path, $fileName)
    {
        $data = $this->importAssetData($path, $fileName);

        if ($view = $this->handleImportErrors($data)) {
            return response()->json($view);
        }

        $csvData = $data['csvData'];

        if ($view = $this->handleCsvColumnErrors($path, $csvData)) {
            return response()->json($view);
        }

        if ($csvData) {

            $validatedData = $this->validateFields($csvData);

            $errors = $validatedData['errors'];
            $csvData = $validatedData['csvData'];

            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            $duplicationValidatedData = $this->checkRepeatUploadData($csvData);
            $csvData = $duplicationValidatedData['csvData'];
            $duplicateErrors = $duplicationValidatedData['errors'];
            $view['duplicate_errors'] = view('assets.partials.upload-duplicate-errors', compact('duplicateErrors'))->render();

            $statusChangeValidatedData = $this->checkCanBeUpdatedToStatus($csvData);
            $statusChangeErrors = $statusChangeValidatedData['errors'];
            $csvData = $statusChangeValidatedData['csvData'];

            // Check the asset is in freezed location or not.
            $locationFreezeValidatedData = $this->validateFromLocationFreeze($csvData);
            $csvData = $locationFreezeValidatedData['csvData'];
            $existingLocationFreezeErrors = $locationFreezeValidatedData['errors'];

            // check the new location is frozen or not
            $newLocationFreezeValidatedData = $this->checkNewLocationFreeze($csvData);
            $csvData = $newLocationFreezeValidatedData['csvData'];
            $newLocationFreezeErrors = $newLocationFreezeValidatedData['errors'];

            $statusChangeErrors = array_merge($statusChangeErrors, $existingLocationFreezeErrors, $newLocationFreezeErrors);
            $view['statusChange_errors'] = view('assets.partials.upload-status-change-errors', compact('statusChangeErrors'))->render();

            $view['data'] = $this->renderCsvData($csvData, $path, $fileName);

            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    /**
     * Generate asset data from the csv array
     * @param  string $item
     * @param  int    $count
     * @return array
     */
    public function generateCSVData(array $item, int $count, $path = null, $fileName = null)
    {
        $assetData    = [];
        foreach ($this->getHeaderMap() as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? trim($item[$fileHeader]) : null;

            if ($dbField == "serial_no") {
                if (isset($item[$fileHeader]) && !empty($item[$fileHeader])) {
                    $assetData = $this->getAdditionalAssetData($assetData, $item[$fileHeader]);
                }
            }
        }

        $assetData = $this->addAdditionalData($assetData);

        if ($fileName && $path) {
            $assetData['filePath'] = $path;
            $assetData['fileName'] = $fileName;
        }

        $count++;
        $assetData['asset_count'] = $count;
        session(['asset_count' => $assetData['asset_count']]);

        return compact('assetData');
    }

    /**
     * Get Additonal Asset Data
     * @param  array  $assetData
     * @param  string $item
     * @return array
     */
    public function getAdditionalAssetData(array $assetData, string $item = "")
    {
        $assetData['asset_id'] = null;
        $assetData['asset_status_id'] = null;
        $assetData['status_name'] = null;

        $asset = Asset::where('serial_no', trim($item))->first();
        if ($asset) {
            $assetData['asset_id'] = $asset->id;
            $assetData['asset_status_id'] = $asset->asset_status_id;
        }

        return $assetData;
    }

    /**
     * Add Additional Data to show in the uploaded view
     * @param array $assetData
     * @return array
     */
    public function addAdditionalData(array $assetData)
    {

        $assetData['current_status_slug'] = isset($assetData['asset_status_id']) ? $this->getStatus($assetData['asset_status_id'])->slug : '';
        $assetData['current_status_name'] = isset($assetData['asset_status_id']) ? $this->getStatus($assetData['asset_status_id'])->name : '';
        $assetData['status_slug'] = $this->getStatusSlug();
        $assetData['status_name'] = $this->getStatusName();

        return $assetData;
    }

    /**
     * Check if the asset can be updated to the status
     * @param  array $csvData
     * @return array
     */
    public function checkCanBeUpdatedToStatus($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        $validData = [];

        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $rowCount = $data['asset_count'] + 1;
            if (isset($data['asset_status_id']) && $data['asset_status_id']) {

                $canUpdatedStatuses = $this->searchFromStatus($this->getStatusSlug());
                $currentStatusSlug = AssetStatus::find($data['asset_status_id'])->slug;

                if (!in_array($currentStatusSlug, $canUpdatedStatuses)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The serial # : ' . $data['serial_no'] . ' cannot be updated to ' . $this->getStatusName() . '.';
                    continue;
                }

                $validData[] = $csvData;
            }
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }

    /**
     * Save Data from the request
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveData(array $inputData, int $i)
    {

        $asset = Asset::where('id', $inputData['asset_id' . $i]);
        //Check if an asset can be updated to the given status
        $check = AssetStatusService::canBeUpdatedTo($inputData['status_name' . $i], $asset);

        if (!$check) {
            return false;
        }

        //Check if an asset is in the freeze location
        $check = AssetStatusService::validateFromLocationFreeze($asset);
        if (!$check) {
            return false;
        }

        $uploadInputData = $this->getData($inputData, $i);

        $uploadInputData['asset_id'] = $inputData['asset_id' . $i];
        $uploadInputData['status_name'] = $inputData['status_name' . $i];

        return UpdateStatusService::updateStatus($uploadInputData, 'bulk_upload');
    }


    /**
     * Save Asset History
     * @param  Asset  $asset
     * @param  array  $inputData
     * @param  int    $i
     * @return void
     */
    public function saveAssetHistory(Asset $asset, array $data)
    {
        $status = optional(AssetStatus::find($data['asset_status_id']))->name;

        $description = AssetHistoryService::getStatusDescription($status, $data, $asset, '');
        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $data['ticket_no'] ?? null,
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'status_updated',
            'comments' => '',
            'old_value' => optional($asset->assetStatus)->name,
            'new_value' => $status,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        $assetHistory = $this->getAdditionalHistoryData($assetHistory, $data, $asset);
        event(new BulkUpdates($assetHistory));
    }

    /**
     * The function checks if the from location of an asset is frozen or not.
     * @param  array $csvData
     * @return array
     */
    public function validateFromLocationFreeze($csvDatas)
    {
        $validData = [];
        $errors = [];
        $rowCount = 1;
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];

            if (isset($data['serial_no']) && $data['serial_no']) {
                $asset = Asset::select('id', 'location_id')->with('location')->where('serial_no', $data['serial_no'])->first();
                if ($asset && ($asset->location && optional($asset->location)->is_freeze == 1)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The asset #' . $data['serial_no'] . ' exists in a frozen location and can\'t be updated.';
                    continue;
                }
                $validData[] = $csvData;
            }
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }

    /**
     * The function checks if a location in a CSV file is frozen for cycle count and returns any
     * errors.
     *
     * @param csvDatas An array of data
     *
     * @return an array of errors.
     */
    public function checkNewLocationFreeze($csvDatas)
    {
        $validData = [];
        $errors = [];
        $rowCount = 1;
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];

            if (isset($data['location_id']) && $data['location_id'] != '') {
                if (Location::select('id', 'is_freeze')->find($data['location_id'])->is_freeze === 1) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : Selected location is frozen for cycle count.';
                    continue;
                }
            }

            $validData[] = $csvData;
        }

        return ['errors' => $errors, 'csvData' => $validData];
    }
}
