<?php

namespace App\Services;

use Facades\App\Repositories\BulkUpload;
use Illuminate\Support\Facades\Validator;
use App\Models\Asset;
use App\Models\AssetStatus;
use Illuminate\Support\Facades\Auth;
use Facades\App\Services\AssetHistory as RepoAssetHistory;
use App\Events\BulkUpdates;

class BulkDamage
{

    /**
     * Imports asset data from a CSV file.
     *
     * @param string $path The file path to the CSV data.
     *
     * @return array An array containing the CSV data and any errors encountered.
     */
    public function importAssetData($path)
    {
        $count = 1;
        $data = $this->getItems($path);
        $items = $data['csvData'];

        $csvData = [
            'error' => $data['error'],
            'csvData' => []
        ];

        if (!empty($items)) {
            foreach ($items as $item) {
                // Skip empty columns.
                if (!count($item)) {
                    continue;
                }

                $csvData['csvData'][] = $this->generateCSVData($item, $count++);
            }
        }
        return $csvData;
    }

    /**
     * Retrieves items from a CSV file.
     *
     * @param string $path The file path to the CSV data.
     *
     * @return array|false An array of items from the CSV file, or false if the file does not exist or is empty.
     */
    public function getItems($path)
    {
        if (!file_exists($path)) {
            return false;
        }

        $items = csv_to_array($path);

        if (empty($items)) {
            return false;
        }

        return $items;
    }

    /**
     * Generates CSV data for an asset.
     *
     * @param array $item  The item data from the CSV file.
     * @param int   $count The current count of processed items.
     *
     * @return array|false An array containing the processed asset data, or false if the item is empty.
     */
    public function generateCSVData($item, $count)
    {
        $headerMap = Config('bulk-upload.damagedAssetData');

        if (empty($item)) {
            return false;
        }

        $assetData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }

            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "serial_no") {
                if (!empty($item[$fileHeader])) {
                    $asset = Asset::where('serial_no', trim($item[$fileHeader]))->first();

                    if ($asset) {
                        $assetData['asset_id'] = $asset->id;
                    }

                    if (!$asset) {
                        $assetData['asset_id'] = null;
                    }
                }
            }
        }

        $assetData['asset_count'] = $count++;
        session(['asset_count' => $assetData['asset_count']]);

        return compact('assetData');
    }

    /**
     * Retrieves relational values for specified fields.
     *
     * @param array  $item       The item data from the CSV file.
     * @param string $dbField    The database field to be processed.
     * @param string $fileHeader The corresponding file header from the CSV.
     *
     * @return mixed The relational value for the field, or false if no value is found.
     */
    public function getRelationalValues($item, $dbField, $fileHeader)
    {
        if ($dbField == "asset_status_id") {
            return BulkUpload::getStatusId($item, $dbField, $fileHeader);
        }

        if ($dbField == "location_id") {
            return BulkUpload::getLocationId($item, $dbField, $fileHeader);
        }

        return false;
    }

    /**
     * Validates the columns of the CSV file against the expected headers.
     *
     * @param string $path    The file path to the CSV data.
     * @param array  $csvData The data extracted from the CSV file.
     *
     * @return string|bool A validation message if there are issues, or true if the columns are valid.
     */
    public function validateCsvColumns($path, $csvData)
    {
        $headerMap = Config('bulk-upload.damagedAssetData');
        $headerKeys = array_values($headerMap);

        $csv = array_map("str_getcsv", file($path, FILE_SKIP_EMPTY_LINES));
        $csvKeys = array_shift($csv);

        if (empty($csvData)) {
            return 'Please make sure the file contains data';
        }

        return $this->checkFileDifference($csvKeys, $headerKeys);
    }

    /**
     * Validates the fields in the CSV data.
     *
     * @param array $csvData The CSV data to be validated.
     *
     * @return array An array of validation errors.
     */
    public function validateFields($csvData)
    {
        $count = 2;

        foreach ($csvData as $data) {
            $errors[] = $this->csvValidator($data['assetData'], $count++);
        }

        return $errors;
    }

    /**
     * Validates the CSV data for an asset.
     *
     * @param array $data  The data to be validated.
     * @param int   $count The line number of the data in the CSV file.
     *
     * @return \Illuminate\Support\MessageBag|null The validation errors, or null if validation passes.
     */
    public function csvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                'serial_no'          => 'required|exists:assets,serial_no',
                'location_id'        => 'required|exists:locations,id',
                'asset_status_id'    => 'required',
            ],
            [
                'serial_no.required'          => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.exists'            => 'Line no ' . $count . ' : The serial # does not exists.',
                'location_id.required'        => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'          => 'Line no ' . $count . ' : The location does not exists.',
                'asset_status_id.required'    => 'Line no ' . $count . ' : The status is required.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Checks for differences between the CSV file columns and the expected header columns.
     *
     * @param array $csvKeys    The columns from the CSV file.
     * @param array $headerKeys The expected header columns.
     *
     * @return string A message indicating any missing or extra columns.
     */
    public function checkFileDifference($csvKeys, $headerKeys)
    {
        if (count($headerKeys) >= count($csvKeys)) {
            $result = array_diff($headerKeys, $csvKeys);
            $messageString = 'Please make sure the column names are spelled correctly. Columns missing - ';
        } else {
            $result = array_diff($csvKeys, $headerKeys);
            $messageString = 'Uploaded file is not match with template file. Extra columns - ';
        }

        $messages = '';

        if (!empty($result)) {
            foreach ($result as $value) {
                $messageString .= $value . ', ';
            }

            $messages = rtrim($messageString, ", ");
        }

        return $messages;
    }

    /**
     * Imports special asset data from a CSV file and updates the assets in the database.
     *
     * @param string $path The file path to the CSV data.
     *
     * @return array An array of error data encountered during the import process.
     */
    public function importAssetSpecialData($path)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $errorData = [];

        if (!empty($items)) {
            foreach ($items as $item) {
                // Skip empty columns.
                if (!count($item)) {
                    continue;
                }

                $csvData = $this->generateSpecialCSVData($item);
                $assetData = $csvData['assetData'];

                if ($csvData['errorData']) {
                    $errorData[] = $csvData['errorData'];
                }

                if (isset($assetData['id'])) {
                    $asset = Asset::with('assetStatus', 'childrenAsset')->where('id', $assetData['id'])->first();

                    if ($asset) {
                        $comments = null;
                        $data['user_id'] = $assetData['user_id'];
                        $data['location_id'] = $assetData['location_id'];
                        $status = AssetStatus::find($assetData['asset_status_id'])->name;
                        $description = RepoAssetHistory::getStatusDescription($status, $data, $asset, $comments);

                        $assetHistory = [
                            'user_id' => Auth::id(),
                            'asset_id' => $asset->id,
                            'action' => 'status_updated',
                            'comments' => $comments,
                            'old_value' => optional($asset->assetStatus)->name,
                            'new_value' => $status,
                            'description' => $description,
                        ];

                        $assetHistory = $this->getHistoryData($assetHistory, $data, $asset);
                        event(new BulkUpdates($assetHistory));

                        $asset->update($assetData);

                        if ($asset->has('childrenAsset')) {
                            foreach ($asset->childrenAsset as $assetChild) {
                                $assetChild->update([
                                    'user_id' => $assetData['user_id'],
                                    'location_id' => $assetData['location_id'],
                                    'asset_status_id' => $assetData['asset_status_id'],
                                ]);
                            }
                        }
                    }
                }
            }
        }

        return $errorData;
    }

    /**
     * Generates special CSV data for an asset.
     *
     * @param array $item The item data from the CSV file.
     *
     * @return array|false An array containing the processed asset data and any error data, or false if the item is empty.
     */
    public function generateSpecialCSVData($item)
    {
        $headerMap = Config('bulk-upload.damagedAssetData');

        if (empty($item)) {
            return false;
        }

        $assetData    = [];
        $errorData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }

            if ($result == false) {
                if ($dbField == "location_id") {
                    $errorData[] = 'Location with room name ' . $item[$fileHeader] . ' does not exist';
                }
            }

            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "serial_no") {
                if (!empty($item[$fileHeader])) {
                    $asset = Asset::where('serial_no', trim($item[$fileHeader]))->first();

                    if ($asset) {
                        $assetData['id'] = $asset->id;
                        $assetData['loaner_retention_date'] = null;
                        $assetData['wipe_confirmation'] = null;
                        $assetData['lost_date'] = null;
                        $assetData['end_of_life_date'] = null;
                        $assetData['user_id'] = null;
                    }

                    if (!$asset) {
                        $errorData[] = 'Asset with serial # ' . $item[$fileHeader] . ' do not exist';
                    }
                }
            }
        }

        return compact('assetData', 'errorData');
    }

    /**
     * Prepares the asset history data for logging changes.
     *
     * @param array $assetHistory The initial asset history data.
     * @param array $assetData    The new asset data.
     * @param Asset $asset        The current asset model.
     *
     * @return array The updated asset history data.
     */
    public function getHistoryData($assetHistory, $assetData, $asset)
    {
        $keyArray = ['loaner_return_date', 'location_id', 'user_id', 'loaner_retention_date', 'asset_status_id'];

        foreach ($keyArray as $key) {
            if (array_key_exists($key, $assetData)) {
                $assetHistory['new_' . $key] = $assetData[$key];
                $assetHistory['old_' . $key] = $asset->$key;
            }
        }

        if (array_key_exists('wipe_confirmation', $assetData)) {
            $assetHistory['wipe_confirmation'] = $assetData['wipe_confirmation'];
        }

        return $assetHistory;
    }
}
