<?php

namespace App\Services\AssetsHealth;

use App\Models\AssetHealthHistory;
use Carbon\Carbon;
use App\Models\AssetHealthReports;
use App\Models\AssetHealthTests;
use App\Models\AssetHealthReportDetail;
use App\Models\AssetHealthCategory;

use App\Repositories\AssetsHealth\AssetsHealthReportRepository;
use App\Services\AssetsHealth\AssetsHealthHistoryService;
use App\Services\AssetsHealth\AssetsHealthDashboardService;
use mikehaertl\wkhtmlto\Pdf;
use Log;
use Illuminate\Support\Facades\File;

class AssetsHealthReportSyncService
{
    /**
     * Create a new AssetsHealthReportSyncService instance.
     *
     * @param AssetsHealthReportRepository $assetHealthReportRepository The repository for asset health reports.
     * @param AssetsHealthHistoryService $assetHealthHistoryService The service for asset health history.
     * @param AssetsHealthDashboardService $assetHealthDashboardService The service for asset health dashboard.
     */
    public function __construct(
        protected AssetsHealthReportRepository $assetHealthReportRepository,
        protected AssetsHealthHistoryService $assetHealthHistoryService,
        protected AssetsHealthDashboardService $assetHealthDashboardService,
    ) {
    }

    /**
     * Synchronize asset health data.
     *
     * @return array The health test reports count.
     */
    public function sync()
    {
        $startTime = now()->format('Y-m-d H:i:s');

        $lastHistory = AssetHealthHistory::orderByDesc('created_at')->first();

        $history = AssetHealthHistory::create([
            'no_of_issues' => 0,
            'started_at' => $startTime,
            'ended_at' => now()
        ]);

        $healthTestReports = $this->saveHealthTestReports($history);

        $history->no_of_issues = $healthTestReports['count'];
        $history->save();

        if ($lastHistory) {
            $syncDate = Carbon::parse($lastHistory->started_at);

            if ($syncDate->isToday()) {
                $lastHistory->delete();
            }
        }

        return $healthTestReports;
    }

    /**
     * Save health test reports for the given asset health history.
     *
     * @param AssetHealthHistory $history The asset health history to save reports for.
     *
     * @return array The count of health test reports saved.
     */
    private function saveHealthTestReports(AssetHealthHistory $history)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $healthTests = AssetHealthTests::all();
        $totalReportsCount = 0;

        foreach ($healthTests as $tests) {
            $testData = $this->getHealthTestFunction($tests->slug);
            $reportsCount = $this->storeTestReports($testData, $tests, $tests->category_id, $history);

            $totalReportsCount += $reportsCount;
        }

        return ['count' => $totalReportsCount];
    }

    /**
     * Store test reports for the given test data, category, and asset health history.
     *
     * @param \Illuminate\Database\Eloquent\Collection|null $testData The test data to store reports for.
     * @param \App\Models\AssetHealthTests $test The health test related to the reports.
     * @param int $categoryId The ID of the category for the reports.
     * @param \App\Models\AssetHealthHistory $history The asset health history related to the reports.
     *
     * @return int The count of test reports stored.
     */
    private function storeTestReports($testData, $tests, $categoryId, $history)
    {
        $testAssetsId = null;

        $testResultCount = 0;
        $ignoredAssetsCount = 0;

        if ($testData && $tests->status) {
            $testAssetsId = $testData->get()->pluck('id');
            $testResultCount = $testData->count();
        }

        $previousReport = AssetHealthReports::where('test_id', $tests->id)->where('category_id', $categoryId)->orderByDesc('id')->first();

        $report = AssetHealthReports::create([
            'test_id' => $tests->id,
            'category_id' => $categoryId,
            'history_id' => $history->id,
            'test_count' => $testResultCount
        ]);

        $reportId = $report->id;

        if ($testAssetsId) {
            $ignoredAssetsCount = $this->storeTestReportDetails($reportId, $testAssetsId, $previousReport);
        }

        $report->update([
            'test_count' => ($testResultCount - $ignoredAssetsCount)
        ]);

        return $testResultCount;
    }

    /**
     * Store test report details for the given report ID, test assets IDs, and previous report.
     *
     * @param int $reportId The ID of the report to store details for.
     * @param \Illuminate\Support\Collection|null $testAssetsId The IDs of the test assets.
     * @param \App\Models\AssetHealthReports|null $previousReport The previous report to check for ignored assets.
     *
     * @return int The count of ignored assets.
     */
    private function storeTestReportDetails($reportId, $testAssetsId, $previousReport)
    {
        $ignoredAssets = [];

        if ($previousReport) {
            $ignoredAssets = AssetHealthReportDetail::where('asset_health_report_id', $previousReport->id)
                ->where('is_ignored', 1)
                ->pluck('asset_id')
                ->toArray();
        }

        foreach ($testAssetsId as $assetId) {
            $isIgnored = in_array($assetId, $ignoredAssets) ? 1 : 0;

            AssetHealthReportDetail::create([
                'asset_health_report_id' => $reportId,
                'asset_id' => $assetId,
                'is_ignored' => $isIgnored
            ]);
        }

        return count($ignoredAssets);
    }

    /**
     * Get the health test function based on the provided key.
     *
     * @param string $key The key to look up in the method map.
     *
     * @return mixed|null The health test function if found, otherwise null.
     */
    private function getHealthTestFunction($key)
    {
        $methodMap = config('asset-health.method_map');

        // Check if the key exists in the method map, then execute the corresponding method
        if (array_key_exists($key, $methodMap)) {
            return $this->assetHealthReportRepository->{$methodMap[$key]}();
        }
    }

    /**
     * Creates a PDF report for asset health data.
     *
     * @return bool True if the PDF report is successfully created, otherwise false.
     */
    public function createReportPdf()
    {
        $assetHealthData = $this->assetHealthDashboardService->getAssetHealthData();

        $assetHealth = [
            'labels' => ['Without Issues', 'With Issues'],
            'datasets' => [
                [
                    'data' => [$assetHealthData, (100 - $assetHealthData)],
                    'backgroundColor' => ['#0da5f9', '#cfccc6'],
                ],
            ],
        ];

        $inputData['dateFrom']  = $this->assetHealthHistoryService->getFromDate();
        $inputData['dateTo'] = $this->assetHealthHistoryService->getToDate();

        $inputData['testId'] = null;

        // Get health report data for errors
        $errorData = $this->getAssetHealthReportData($inputData, 'errors');

        // Get health report data for warnings
        $warningData = $this->getAssetHealthReportData($inputData, 'warnings');

        // Get health report data for notices
        $noticeData = $this->getAssetHealthReportData($inputData, 'notices');

        $topIssues = $this->assetHealthDashboardService->getTopAssetHealthIssues()->get();

        $allCategories = $this->assetHealthDashboardService->getCategories();

        foreach ($allCategories as $category) {
            $testReports[$category->slug] = $this->assetHealthDashboardService->getHealthReportsByTests($category->id)->get();
        }

        // Render PDF pages
        $pages = [];
        $pages[] = view('assets-health.pdf.pdf1')->render();
        $pages[] = view('assets-health.pdf.pdf2', compact('assetHealth', 'errorData'))->render();
        $pages[] = view('assets-health.pdf.pdf3', compact('warningData', 'noticeData'))->render();
        $pages[] = view('assets-health.pdf.pdf4', compact('topIssues', 'testReports'))->render();
        $pages[] = view('assets-health.pdf.pdf5', compact('testReports'))->render();

        // Initialize PDF
        $pdf = new Pdf([
            'no-outline',         // Make Chrome not complain
            'margin-top'    => 0,
            'margin-bottom' => 0
        ]);

        // Add pages to PDF
        foreach ($pages as $page) {
            $pdf->addPage($page);
        }

        $pdf->setOptions(['javascript-delay' => 5000]);

        $lastHistory = AssetHealthHistory::orderByDesc('created_at')->first();

        $reportDirectory = public_path('reports/' . $lastHistory->created_at->format('Y-m-d'));

        if (!File::exists($reportDirectory)) {
            // Attempt to create directory
            if (!File::makeDirectory($reportDirectory, 0755, true)) {
                // Directory creation failed, handle the error
                Log::error("Failed to create directory: $reportDirectory");

                return false;
            }
        }
        // Save the PDF
        if (!$pdf->saveAs($reportDirectory . '/report.pdf')) {
            $error = $pdf->getError();

            Log::error('PDF Error: ' . $error);

            return false;
        }

        return true;
    }

    /**
     * Retrieves asset health report data based on input data and category.
     *
     * @param array $inputData The input data for generating the report.
     * @param string $categorySlug The slug of the category for which to retrieve the report data.
     *
     * @return array The asset health report data.
     */
    private function getAssetHealthReportData($inputData, $categorySlug)
    {
        $inputData['categoryId'] = AssetHealthCategory::whereSlug($categorySlug)->first()->id;

        return $this->assetHealthHistoryService->assetsHealthReportData($inputData)['datas'];
    }
}
