<?php

namespace App\Services\AssetsHealth;

use App\Models\AssetHealthHistory;
use App\Models\AssetHealthReportDetail;
use App\Models\AssetHealthTests;

class AssetsHealthExportService
{
    /**
     * Retrieve test reports data.
     *
     * This method retrieves the test reports data for the latest asset health history.
     *
     * @return \Illuminate\Database\Eloquent\Builder The query builder for test reports data.
     */
    public function getTestReportsData()
    {
        $latestAssetHealthHistory = AssetHealthHistory::orderBy('created_at', 'desc')->first();

        return AssetHealthReportDetail::with(['asset.assetType','assetHealthReport'])
        ->join('asset_health_reports', 'asset_health_report_details.asset_health_report_id', 'asset_health_reports.id')
        ->where('is_ignored', 0)
        ->where('asset_health_reports.history_id', $latestAssetHealthHistory->id);
    }

    /**
     * Prepare data for downloading test reports.
     *
     * This method prepares data for downloading test reports by iterating through each test detail.
     *
     * @param \Illuminate\Support\Collection $testDetails The collection of test details.
     * @param int $start The starting index.
     * @param array $data The data array.
     *
     * @return array The prepared data for downloading test reports.
     */
    public function getTestReportDownloadData($testDetails, $start, $data)
    {
        $parentIndex = $start;
        $allTests = AssetHealthTests::all();

        foreach ($testDetails as $test) {
            $parentIndex++;
            $nestedData = $this->getDownloadNestedData($test, $allTests);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Prepare nested data for downloading test reports.
     *
     * This method prepares nested data for downloading test reports.
     *
     * @param mixed $report The report data.
     * @param \Illuminate\Support\Collection $allTests The collection of all tests.
     *
     * @return array The prepared nested data for downloading test reports.
     */
    public function getDownloadNestedData($report,$allTests)
    {
        $nestedData['Serial Number']  = $report->asset ? $report->asset->serial_no : "";

        foreach($allTests as $tests){
            $nestedData[$tests->test_name]  = ($report->assetHealthReport->test_id == $tests->id) ? 1 : 0;
        }

        return $nestedData;
    }

    /**
     * Export the provided data to a CSV file.
     *
     * This method exports the provided data to a CSV file and streams it as a response.
     * If the provided data is empty or does not contain the expected structure, an error response is returned.
     *
     * @param array $datas The data to be exported.
     *
     * @return \Illuminate\Http\Response The CSV file stream response.
     */
    public function export($datas)
    {
        // Check if $datas is empty or does not contain the expected structure
        if (empty($datas) || !isset($datas[0][0])) {
            return response()->json(['error' => 'Invalid data format'], 400);
        }

        // Set headers for CSV file
        $headers = [
            "Content-type"        => "text/csv",
            "Content-Disposition" => "attachment; filename=file.csv",
            "Pragma"              => "no-cache",
            "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
            "Expires"             => "0"
        ];

        // Process the data to merge and prepare for CSV export
        $processedData = [];

        // Process each data item
        foreach ($datas as $dataItems) {
            foreach ($dataItems as $data) {
                $serial = $data['Serial Number'];
                unset($data['Serial Number']);

                // Merge data with the same serial number
                if (!isset($processedData[$serial])) {
                    $processedData[$serial] = $data;
                } else {
                    foreach ($data as $key => $value) {
                        $processedData[$serial][$key] = max($processedData[$serial][$key] ?? 0, $value);
                    }
                }
            }
        }

        /// Prepare data array with serial number and headers for CSV file
        $dataArray = array_map(function ($key, $value) {
            return array_merge(['Serial #' => $key], $value);
        }, array_keys($processedData), $processedData);

        // Stream the CSV file
        $callback = function () use ($dataArray) {
            $file = fopen('php://output', 'w');
            fputcsv($file, array_keys($dataArray[0]));

            foreach ($dataArray as $data) {
                fputcsv($file, $data);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
