<?php

namespace App\Services\AssetsHealth;

use App\Models\AssetHealthHistory;
use App\Models\AssetHealthReports;
use App\Repositories\AssetsHealth\AssetsHealthDashboardRepository;

class AssetsHealthDashboardService
{

    /**
     * Constructor for initializing the AssetsHealthDashboardRepository.
     *
     * @param AssetsHealthDashboardRepository $assetsHealthDashboardRepository The repository for assets health dashboard
     */
    public function __construct(protected AssetsHealthDashboardRepository $assetsHealthDashboardRepository)
    {
    }

    /**
     * Retrieve asset health histories data for generating graphs.
     *
     * @return array The array containing the formatted graph data and week names.
     */
    public function getHistoriesData()
    {
        $fromDate = now()->subDays(84)->startOfWeek();
        $toDate = now();

        $resultData = $this->assetsHealthDashboardRepository->getAllAssetHealthHistories(
            $fromDate,
            $toDate
        );

        $weeks = get_asset_health_week_range_count($fromDate, $toDate);

        $datas = $this->getFormattedGraphData($resultData, $weeks);

        $weekNames = collect($weeks)->flatten()->toArray();

        return compact('datas', 'weekNames');
    }

    /**
     * Format asset health history data for graph display.
     *
     * @param  mixed  $assetHistories  The array containing asset health history data.
     * @param  array  $weeks  The array containing week numbers and names.
     *
     * @return array The formatted data for graph display.
     */
    private function getFormattedGraphData($assetHistories, $weeks)
    {
        $formattedData = [];

        $assetHealthHistoryData = $assetHistories->toArray();

        foreach ($weeks as $weekNumber => $weekName) {
            $totalIssues = 0;

            if (in_array($weekNumber, array_column($assetHealthHistoryData, 'week'))) {
                $totalIssues = array_column($assetHealthHistoryData, 'count', 'week')[$weekNumber];
            }

            $formattedData[] = [
                'weekDate' => $weekName,
                'weekDay' => $weekNumber,
                'totalIssues' => $totalIssues,
            ];
        }

        return $formattedData;
    }

    /**
     * Retrieve all asset health categories.
     *
     * @return \Illuminate\Database\Eloquent\Collection The collection containing all asset health categories.
     */
    public function getCategories()
    {
        return $this->assetsHealthDashboardRepository->getAllAssetHealthCategories();
    }

    /**
     * Retrieve an asset health category by its slug.
     *
     * @param  string  $categorySlug The slug of the asset health category.
     *
     * @return mixed|null The asset health category if found, otherwise null.
     */
    public function getCategoryBySlug($categorySlug)
    {
        return $this->assetsHealthDashboardRepository->getAssetHealthCategoryBySlug($categorySlug);
    }

    /**
     * Retrieve all asset health categories with associated tests.
     *
     * @return \Illuminate\Support\Collection The collection containing all asset health categories with associated tests.
     */
    public function getCategoriesWithTests()
    {
        return $this->assetsHealthDashboardRepository->getAllAssetHealthTests();
    }

    /**
     * Retrieve the count of asset health tests for each category.
     *
     * @return \Illuminate\Support\Collection The collection containing the count of asset health tests for each category.
     */
    public function getCategoryTestCount()
    {
        return $this->assetsHealthDashboardRepository->getCategoryWiseAssetHealthCount();
    }

    /**
     * Retrieve the top asset health issues.
     *
     * @return \Illuminate\Database\Eloquent\Collection The collection containing the top asset health issues.
     */
    public function getTopAssetHealthIssues()
    {
        return $this->assetsHealthDashboardRepository->getTopAssetHealthReportIssues();
    }

    /**
     * Retrieve asset health reports by tests for a specific category.
     *
     * @param  int  $categoryId The ID of the category.
     *
     * @return \Illuminate\Database\Eloquent\Collection The collection containing asset health reports by tests for the specified category.
     */
    public function getHealthReportsByTests($categoryId)
    {
        return $this->assetsHealthDashboardRepository->getAssetHealthReportsByTests($categoryId);
    }

    /**
     * Retrieve the asset health test detail by report ID.
     *
     * @return mixed The asset health test detail.
     */
    public function getTestDetailById($reportId)
    {
        return $this->assetsHealthDashboardRepository->getAssetHealthTestDetailById($reportId);
    }

    /**
     * Retrieve the ignored asset health test detail by report ID.
     *
     * @param  int  $reportId The ID of the asset health report.
     *
     * @return mixed The ignored asset health test detail.
     */
    public function getIgnoredTestDetailById($reportId)
    {
        return $this->assetsHealthDashboardRepository->getAssetHealthIgnoredTestDetailById($reportId);
    }


    /**
     * Retrieve the asset health test report details.
     *
     * @param  int  $reportId The ID of the asset health report.
     *
     * @return array The array containing test details and their count.
     */
    public function getTestReportDetails($reportId)
    {
        $testDetails = $this->assetsHealthDashboardRepository->getAssetHealthReportDetails($reportId);

        $count      = $testDetails->count();
        $testDetails     = $this->getOutputData($testDetails);

        return compact('testDetails', 'count');
    }

    /**
     * Retrieve the asset health test report details for Excel export.
     *
     * @param  int  $reportId The ID of the asset health report.
     *
     * @return \Illuminate\Database\Eloquent\Collection The collection containing test report details for Excel export.
     */
    public function getTestReportDetailsForExcel($reportId)
    {
        return $this->assetsHealthDashboardRepository->getAssetHealthReportDetails($reportId)->orderBy('id', 'ASC')->get();
    }


    /**
     * Retrieve the ignored asset health test report details.
     *
     * @param  int  $reportId The ID of the asset health report.
     *
     * @return array The array containing ignored test details and their count.
     */
    public function getIgnoredTestReportDetails($reportId)
    {
        $testDetails = $this->assetsHealthDashboardRepository->getAssetHealthReportDetails($reportId, true);

        $count      = $testDetails->count();
        $testDetails     = $this->getOutputData($testDetails);

        return compact('testDetails', 'count');
    }

    /**
     * Retrieve formatted output data for pagination.
     *
     * @param  mixed  $healthTest The health test data.
     *
     * @return \Illuminate\Database\Eloquent\Collection The collection containing formatted output data.
     */
    public function getOutputData($healthTest)
    {
        $start = request('start');
        $limit = request('length');

        if ($limit != -1) {
            $healthTest = $healthTest->offset($start)
                ->limit($limit);
        }

        $healthTest->orderBy('id', 'ASC');

        return $healthTest->get();
    }

    /**
     * Retrieve report data for a test.
     *
     * @param  mixed  $testDetails The test details.
     * @param  int    $start       The start index for pagination.
     * @param  array  $data        The data array to store the report data.
     * @param  mixed  $testData    The test data.
     *
     * @return array The updated data array containing the report data.
     */
    public function getReportData($testDetails, $start, $data, $testData)
    {
        $parentIndex = $start;

        foreach ($testDetails as $test) {
            $parentIndex++;
            $nestedData = $this->getNestedData($test, $parentIndex, $testData);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieve ignored report data for a test.
     *
     * @param  mixed  $testDetails The ignored test details.
     * @param  int    $start       The start index for pagination.
     * @param  array  $data        The data array to store the ignored report data.
     * @param  mixed  $testData    The test data.
     *
     * @return array The updated data array containing the ignored report data.
     */
    public function getIgnoredReportData($testDetails, $start, $data, $testData)
    {
        $parentIndex = $start;

        foreach ($testDetails as $test) {
            $parentIndex++;
            $nestedData = $this->getIgnoredNestedData($test, $parentIndex, $testData);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Get nested data for the report.
     *
     * @param  mixed  $reportData The report data.
     * @param  int  $index The index for the report data.
     * @param  mixed  $testData The test data.
     *
     * @return array The nested data for the report.
     */
    public function getNestedData($reportData, $index, $testData)
    {
        return $this->getCommonNestedData($reportData, $index, $testData);
    }

    /**
     * Get nested data for the report in Excel format.
     *
     * @param  mixed  $reportData The report data.
     * @param  int  $index The index for the report data.
     * @param  mixed  $testData The test data.
     *
     * @return array The nested data for the report in Excel format.
     */
    public function getNestedDataForExcel($reportData, $index, $testData)
    {
        return $this->getCommonNestedData($reportData, $index, $testData, false, true);
    }

    /**
     * Get nested data for ignored report in the dashboard.
     *
     * @param  mixed  $reportData The report data.
     * @param  int  $index The index for the report data.
     * @param  mixed  $testData The test data.
     *
     * @return array The nested data for ignored report in the dashboard.
     */
    public function getIgnoredNestedData($reportData, $index, $testData)
    {
        return $this->getCommonNestedData($reportData, $index, $testData, true);
    }

    /**
     * Get nested data for the report.
     *
     * @param  mixed  $reportData The report data.
     * @param  int  $index The index for the report data.
     * @param  mixed  $testData The test data.
     * @param  bool  $ignored Whether the report is ignored or not.
     * @param  bool  $forExcel Whether the data is for Excel format or not.
     *
     * @return array The nested data for the report.
     */
    private function getCommonNestedData($reportData, $index, $testData, $ignored = false, $forExcel = false)
    {
        $testSlug = $testData->slug ?? 'default';

        $assetLink = $this->getAssetDetailsRoute($reportData->asset);

        $returnFields = config('asset-health.datatable_columns')[$testSlug];

        $nestedData = [];

        foreach ($returnFields as $key => $label) {
            $nestedData[$key] = $this->getNestedDataForField($reportData, $key, $assetLink, $forExcel);
        }

        if (!$forExcel) {
            $nestedData['ignore'] = $this->getIgnoreButton($ignored, $reportData->id);
        }

        return $nestedData;
    }

    /**
     * Get the value for the specified field in the nested data.
     *
     * @param  mixed   $reportData The report data.
     * @param  string  $key        The key corresponding to the field.
     * @param  string  $assetLink  The asset link.
     * @param  bool    $forExcel   Flag indicating whether the data is for Excel.
     *
     * @return string  The value for the specified field.
     */
    private function getNestedDataForField($reportData, $key, $assetLink, $forExcel = false)
    {
        $result = '';

        switch ($key) {
            case 'id':
                $result = $reportData->id;
                break;
            case 'asset_type':
                $result = optional($reportData->asset)->assetType->name ?? '';
                break;
            case 'asset_tag':
                $result = optional($reportData->asset)->asset_tag ?? '';
                break;
            case 'serial_no':
                $result = $this->getSerialNo($reportData->asset, $assetLink, $forExcel);
                break;
            case 'hardware_standard':
                $result = optional(optional($reportData->asset)->makeAndModel)->makeModelName ?? '';
                break;
            case 'tech_specs':
                $result = optional(optional($reportData->asset)->technicalSpec)->details ?? '';
                break;
            case 'user_location':
                $userName = optional(optional($reportData->asset)->user)->userName ?? '';
                $roomName = optional(optional($reportData->asset)->location)->room_name ?? '';
                $result = $userName ?: $roomName;
                break;
            case 'asset_status':
                $result = optional(optional($reportData->asset)->assetStatus)->name ?? '';
                break;
            case 'last_modified_date':
                $result = optional(optional($reportData->asset)->latestAssetHistory)->updated_at ?? '';
                break;
            case 'last_modified_by':
                $result = optional(optional(optional($reportData->asset)->latestAssetHistory)->user)->userName ?? '';
                break;
            case 'last_seen':
                $result = optional($reportData->asset)->lastSeen ?? '';
                break;
            case 'warranty_end_date':
                $result = optional($reportData->asset)->warranty_end_date ?? '';
                break;
            case 'asset_age':
                $result = optional($reportData->asset)->asset_age ?? '';
                break;
            default:
                $result = '';
        }

        return $result;
    }

    /**
     * Get the serial number for the asset, optionally with a link.
     *
     * @param  mixed   $asset       The asset data.
     * @param  string  $assetLink   The asset link.
     * @param  bool    $forExcel    Flag indicating whether the data is for Excel.
     *
     * @return string  The serial number with or without a link.
     */
    private function getSerialNo($asset, $assetLink, $forExcel = false)
    {
        if (!$asset) {
            return '';
        }

        $serialNo = $asset->serial_no;

        if ($forExcel) {
            return $serialNo;
        }

        return $assetLink ? "<a href=\"$assetLink\" target=\"__blank\">$serialNo</a>" : $serialNo;
    }

    /**
     * Get the ignore button HTML link based on whether the report is ignored or not.
     *
     * @param  bool    $ignored    Flag indicating whether the report is ignored.
     * @param  int     $reportId   The ID of the report.
     * @return string  The HTML link for ignoring or undoing ignore of the report.
     */
    private function getIgnoreButton($ignored, $reportId)
    {
        $ignoreLink = $this->getIgnoreLink($ignored, $reportId);
        $action = $ignored ? 'Undo ' : '';

        return "<a class=\"toggle-ignore-btn\" href=\"$ignoreLink\">$action" . "Ignore</a>";
    }

    /**
     * Get the route URL for ignoring or undoing ignore of a report.
     *
     * @param  bool    $ignored    Flag indicating whether the report is ignored.
     * @param  int     $reportId   The ID of the report.
     *
     * @return string  The route URL for ignoring or undoing ignore of the report.
     */
    private function getIgnoreLink($ignored, $reportId)
    {
        $routeName = $ignored ? 'assets-health-test-detail-undo-ignore' : 'assets-health-test-detail-ignore';

        return route($routeName, $reportId);
    }

    /**
     * Get the asset details route based on the asset type.
     *
     * @param  mixed  $data The asset data.
     *
     * @return string The asset details route.
     */
    public function getAssetDetailsRoute($data)
    {
        if ($data) {
            if ($data->assetType->hasRelationToSlug('mobile_assets')) {
                $assetLink = route('mobile-assets.show', $data->id);
            } elseif ($data->assetType->hasRelationToSlug('av_asset')) {
                $assetLink = route('av-assets.show', $data->id);
            } elseif ($data->assetType->hasRelationToSlug('network_asset')) {
                $assetLink = route('network-assets.show', $data->id);
            } elseif ($data->assetType->hasRelationToSlug('research_asset')) {
                $assetLink = route('research-assets.show', $data->id);
            } else {
                $assetLink = route('assets.show', $data->id);
            }
        } else {
            $assetLink = '#';
        }

        return $assetLink;
    }

    /**
     * Get the export data for the asset health test detail.
     *
     * @param  int  $testReportId  The ID of the asset health test report.
     *
     * @return mixed  The export data for the asset health test detail.
     */
    public function getTestDetailExport($testReportId)
    {
        return $this->assetsHealthDashboardRepository->getAssetHealthReportDetails($testReportId);
    }

    /**
     * Get the export data for the asset health test detail.
     *
     * @param  array  $testDetails  The details of the asset health test.
     * @param  int    $start        The starting index for the data.
     * @param  array  $data         The existing export data.
     * @param  mixed  $testData     The test data.
     *
     * @return array  The updated export data for the asset health test detail.
     */
    public function getTestDetailExportData($testDetails, $start, $data, $testData)
    {
        $parentIndex = $start;

        foreach ($testDetails as $key => $test) {
            $parentIndex++;

            $nestedData = $this->getExportNestedData($test, $parentIndex, $testData);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Get the nested data for export based on the report data.
     *
     * @param  mixed  $reportData  The report data.
     * @param  int  $parentIndex  The parent index for the report data.
     * @param  mixed  $testData  The test data.
     *
     * @return array  The nested data for export.
     */
    public function getExportNestedData($reportData, $parentIndex, $testData)
    {
        $testSlug = $testData->slug ?? 'default';

        $returnFields = config('asset-health.datatable_columns')[$testSlug];

        unset($returnFields['id']);

        foreach ($returnFields as $key => $label) {
            $nestedData[$label] = $this->getNestedDataForField($reportData, $key, '', true);
        }

        return $nestedData;
    }

    /**
     * Export data to CSV format.
     *
     * @param array $datas The data to be exported.
     *
     * @return mixed The exported CSV data.
     */
    public function export($datas)
    {
        return exportToCsv($datas);
    }

    /**
     * Get the percentage of assets with no health issues.
     *
     * This method calculates the percentage of assets with no health issues
     * based on the data obtained from the assets health dashboard repository.
     *
     * @return int The percentage of assets with no health issues.
     */
    public function getAssetHealthData()
    {
        $assetsWithIssues = $this->assetsHealthDashboardRepository->getIssuesAssetsCount();


        $errorCount = $assetsWithIssues['issueAssetsCount']['errors'] ?? 0;
        $warningCount = $assetsWithIssues['issueAssetsCount']['warnings'] ?? 0;

        $totalCount = $assetsWithIssues['totalAssetsCount'];

        if ($totalCount == 0) {
            return 100;
        }

        $errorWeight = .70;
        $warningWeight = .30;

        $weightedScore = ($errorCount * $errorWeight) + ($warningCount * $warningWeight);

        $normalizedScore = ($weightedScore/($totalCount * $errorWeight)) * 100;


        return 100 - floor($normalizedScore);
    }

    /**
     * Retrieve scanned assets data.
     *
     * This method fetches scanned assets data from the assets health dashboard repository.
     *
     * @return mixed Scanned assets data retrieved from the repository.
     */
    public function getscannedAssetsData()
    {
        return $this->assetsHealthDashboardRepository->getAssetHealthscannedAssetsData();
    }

    /**
     * Toggle the ignoring status of a test detail by its ID.
     *
     * This method delegates the task of toggling the ignoring status of an asset health test detail to the repository.
     *
     * @param int $testDetailId The ID of the test detail to toggle ignoring.
     * @param bool $ignoreStatus The status indicating whether to ignore or undo ignoring the test detail.
     *
     * @return mixed The result of toggling the ignoring status of the test detail, returned by the repository.
     */
    public function toggleIgnoreTestDetailById($testDetailId, $ignoreStatus)
    {
        return $this->assetsHealthDashboardRepository->toggleIgnoreAssetHealthTestDetailById($testDetailId, $ignoreStatus);
    }

    /**
     * Retrieve health report by test slug.
     *
     * This method retrieves the health report by the specified test slug for the latest asset health history.
     *
     * @param string $testSlug The slug of the test.
     *
     * @return \Illuminate\Database\Eloquent\Builder The query builder for the health report.
     */
    public function getHealthReportByTestSlug($testSlug)
    {
        $latestAssetHealthHistory = AssetHealthHistory::orderBy('created_at', 'desc')->first();

        return AssetHealthReports::with('assetHealthTest')->whereRelation('assetHealthTest', 'slug', '=', $testSlug)
        ->where('asset_health_reports.history_id', $latestAssetHealthHistory->id);
    }

    /**
     * Retrieve the latest asset health history.
     *
     * This method fetches the most recent asset health history record.
     *
     * @return mixed The latest asset health history record.
     */
    public function getLatestAssetHealthHistory()
    {
        return $this->assetsHealthDashboardRepository->getLatestAssetHealthHistory();
    }
}
