<?php

namespace App\Services;

use App\Models\Asset;
use App\Models\AssetStatus as Status;
use App\Models\AssetType;
use App\Models\UserType;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

/*
|--------------------------------------------------------------------------------------------
| This is the base class that calls the different classes in StatusUpdateConditions
| The conditions for each status update is stored in 2 places
| 1. All interfaces implementing App\Services\StatusUpdateConditions\StatusAllowedForAsset
| 2. config/status.php
|--------------------------------------------------------------------------------------------
*/

class AssetStatus
{

	/**
	 * The status that should be set to a newly created asset
	 *
	 * @return string The status slug
	 */
	public function whenCreatingAsset()
	{
		return 'brand_new'; //status that should be set to a newly created asset
	}

	/**
	 * Get the status for a newly created asset.
	 *
	 * @param Asset $asset
	 * @return Status
	 */
	public function getStatusForNewAsset()
	{
		return Status::where('slug', $this->whenCreatingAsset())->first();
	}


	/**
	 * Retrieves the class name for a status update based on the provided slug.
	 *
	 * @param string $slug The slug used to generate the status class name
	 * @return string The class name for the status update
	 */
	private function getStatusUpdateClassName($slug)
	{
		$status = Str::ucfirst($slug);
		$statusClass = str_replace(' ', '', $status);
		$statusClass = str_replace('/', '', $statusClass);
		$statusClass = str_replace('-', '', $statusClass);
		$slug = str_slug($slug, '_');
		$namespace = "\App\Services\StatusUpdateConditions\\";
		$className = $namespace . "Status$statusClass"; //eg. StatusNew

		if (!class_exists($className)) {
			$className = $namespace . "StatusCommon";
		}
		return $className;
	}

	/**
	 * Check if an asset can be updated to a given status.
	 *
	 * The conditions for the update are checked in the following order:
	 * 1. Whether the asset can be updated from the current status.
	 * 2. Whether the asset can be updated to the given status.
	 * 3. Whether the current status is allowed to be updated to the given status.
	 *
	 * @param string $statusSlug The slug of the status to be updated to.
	 * @param Asset $asset The asset model instance.
	 * @return boolean Whether the asset can be updated to the given status.
	 */
	public function canBeUpdatedTo($statusSlug, $asset)
	{
		$currentStatusClassName = $this->getStatusUpdateClassName(optional($asset->first()->assetStatus)->name);
		$currentStatusClass = new $currentStatusClassName;

		// Checks the conditions for updating from the current status
		if (!$currentStatusClass->canbeUpdatedFromThisStatus($asset)) {
			return false;
		}

		$toStatusClassName = $this->getStatusUpdateClassName($statusSlug);

		$toStatusClass = new $toStatusClassName;
		if (!$toStatusClass->allowedForAsset($asset)) {
			return false;
		}

		$asset = $asset->first();
		if ($asset->asset_status_id) {
			$currentStatusSlug = Status::find($asset->asset_status_id)->slug;
			$statusSlug = str_slug($statusSlug, '_');
			if (!in_array($statusSlug, config('asset-status.' . $currentStatusSlug))) {
				return false;
			}
		}

		return true;
	}

	public function availableStatusesForUpdate($statusSlug, $assetId)
	{
		$allowedStatusForAsset = config('asset-status.' . $statusSlug);
		if ($statusSlug == 'legal_hold') {
			$userTypeId = User::find(Auth::id())->user_type_id;
			$userTypes = ['Administrator', 'Super Admin'];
			$userTypeName = UserType::find($userTypeId)->name;
			if (!in_array($userTypeName, $userTypes)) {
				$allowedStatusForAsset = [];
			}
		}

		if (request('general_type') == 'av' || request('general_type') == 'network') {
			if (($key = array_search('assigned', $allowedStatusForAsset)) !== false) {
				unset($allowedStatusForAsset[$key]);
			}
		}

		/* below code commented out for optimisation */
		// $allowedStatuses = [];
		// if(isset($allowedStatusForAsset)){
		// 	foreach($allowedStatusForAsset as $allowedStatus) {
		// 	$asset = Asset::with('assetStatus', 'location.locationType')->where('id', $assetId);
		// 	$statusName = Status::getNameFromSlug($allowedStatus)->first()->name;
		// 		if($this->canBeUpdatedTo($statusName, $asset)) {
		// 			$allowedStatuses[] = $allowedStatus;
		// 		}
		//     }
		// }

		// return Status::whereIn('slug', $allowedStatuses)->orWhere('slug', $statusSlug)->get();
		// dd($allowedStatusForAsset);
		// return Status::whereIn('slug', ["assigned", "installed", "loaned", "like_new"])->orWhere('slug', $statusSlug)->get();
		if (!is_array($allowedStatusForAsset)) {
			$allowedStatusForAsset = [];
		}
		return Status::whereIn('slug', $allowedStatusForAsset)->orWhere('slug', $statusSlug)->get();
	}

	public function validateStatusUpdate($from, $to)
	{
		$availableStatuses = config('asset-status.' . $from);

		return in_array($to, $availableStatuses);
	}

	public function canBeMoved($asset)
	{
		return $asset->assetStatus->has_location;
	}

	public function canBeReturned($asset)
	{
		return $asset->assetStatus->has_user;
	}

	/**
	 * The function checks if the from location of an asset is frozen or not.
	 * 
	 * @param asset 
	 * 
	 * @return true or false
	 */
	public function validateFromLocationFreeze($asset)
	{
		$asset = $asset->first();

		return $asset->location ? (optional($asset->location)->is_freeze == 0 ? true : false) : true;
	}
}
