<?php

namespace App\Services;

use App\Models\Asset;
use App\User;
use App\Models\AssetHistory as AssetHistoryModel;
use App\Models\Location;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\Events\BulkUpdates;

class AssetHistory
{
	protected $actions = [
		'created'                                         =>  'warning',
		'linked'                                          => 'primary',
		'moved'                                           => 'info',
		'returned'                                        => 'danger',
		'status_updated'                                  => 'success',
		'assetTag_updated'                                => 'update',
		'loanerReturnDate_updated'                              => 'return',
		'carrier_updated'                                 => 'operate',
		'admin_created'                                   => 'admin-created',
		'location_updated'                                => 'warning',
		'role_created'                                    => 'primary',
		'accesslevel_created'                             => 'info',
		'hardware_standard_created'                       => 'danger',
		'password_changed'                                => 'success',
		'assetType_created'                               => 'update',
		'assetAttribute_created'                          => 'return',
		'mobileCarrier_created'                           => 'operate',
		'locationType_created'                            => 'admin-created',
		'region_created'                                  => 'warning',
		'region_updated'                                  => 'primary',
		'region_deleted'                                  => 'info',
		'locationType_updated'                            => 'danger',
		'locationType_deleted'                            => 'success',
		'mobileCarrier_updated'                           => 'update',
		'mobileCarrier_deleted'                           => 'return',
		'assetAttribute_updated'                          => 'operate',
		'assetAttribute_deleted'                          => 'admin-created',
		'assetType_updated'                               => 'warning',
		'assetType_deleted'                               => 'primary',
		'admin_updated'                                   => 'info',
		'admin_deleted'                                   => 'danger',
		'hardware_standard_updated'                       => 'success',
		'hardware_standard_deleted'                       => 'update',
		'manufacturer_created'                            => 'return',
		'manufacturer_updated'                            => 'operate',
		'manufacturer_deleted'                            => 'admin-created',
		'siteCode_created'                                => 'warning',
		'siteCode_deleted'                                => 'primary',
		'siteCode_updated'                                => 'info',
		'unlink'                                          => 'danger',
		'leaseEndDate_updated'                            => 'success',
		'assetAttributes_changed'                         => 'update',
		'location_added'                                  => 'return',
		'user_added'                                      => 'operate',
		'user_updated'                                    => 'operate',
		'serialNo_updated'                                => 'admin-created',
		'imported_by_vendor'                              => 'success',
		'certificate_uploaded'                            => 'success',
		'asset_received'                                  => 'success',
		'software_license_category_created'               => 'danger',
		'software_license_category_updated'               => 'success',
		'software_license_category_deleted'               => 'update',
		'software_license_manufacturer_created'           => 'danger',
		'software_license_manufacturer_updated'           => 'success',
		'software_license_manufacturer_deleted'           => 'update',
		'software_license_subscription_created'           => 'danger',
		'software_license_subscription_updated'           => 'success',
		'software_license_subscription_deleted'           => 'update',
		'vendor_created'                                  => 'danger',
		'vendor_updated'                                  => 'primary',
		'vendor_deleted'                                  => 'update',
		'auto_assigned'                                   => 'primary',
	];

	/**
	 * Returns the history description
	 * @param mixed $status
	 * @param mixed $data
	 * @param mixed $asset
	 * @param null $comments
	 *
	 * @return [type]
	 */
	public function getStatusDescription($status, $data, $asset, $comments = null)
	{
		$namespace = "\App\Services\AssetHistoryDescriptions\\";

		$statusClass = str_replace(' ', '', trim($status));
		$statusClass = str_replace('/', '', $statusClass);
		$statusClass = str_replace('-', '', $statusClass);


		$className = $namespace . "HistoryFor" . $statusClass;
		if (!class_exists($className)) {
			$className = $namespace . "HistoryForCommon";
		}

		$class = new $className;
		return $class->description($status, $data, $asset, $comments);
	}

	public function getLocationDescription($locationSet, $request, $newLocationSet, $comments = null)
	{
		$namespace = "\App\Services\AssetHistoryDescriptions\\";

		$statusClass = 'Location';
		$className = $namespace . "HistoryForLocation";

		$class = new $className;
		return $class->description($locationSet, $request, $newLocationSet, $comments = null);
	}

	/**
	 * Returns old user or location
	 * @param mixed $asset
	 *
	 * @return [type]
	 */
	public function getOldLocationOrUsername($asset)
	{

		if ($asset->assetStatus->has_location == 1) {
			return optional($asset->location)->room_name;
		}

		$olduser = User::select('first_name', 'last_name')->where('id', $asset->user_id)->first();
		return ($olduser != null) ? ($olduser->first_name . ' ' . $olduser->last_name) : null;
	}

	/**
	 * Returns to location or user for asset history descreption creation.
	 * @param mixed $asset
	 *
	 * @return [type]
	 */
	public function getToLocationOrUsername($data)
	{

		if (isset($data['location_id']) && $data['location_id']) {
			$toLocation = Location::select('room_name')->where('id', $data['location_id'])->first();
			$toLocationName = ($toLocation != null) ? (' to ' . $toLocation->room_name) : " ";
			return $toLocationName;
		}

		if (isset($data['user_id']) && $data['user_id']) {
			$toUser = User::select('first_name', 'last_name', 'email')->where('id', $data['user_id'])->first();
			$toUsername = ($toUser != null) ? (' to ' . $toUser->userName) : " ";

			return $toUsername;
		}

		return '';
	}

	public function getActions()
	{
		return $this->actions;
	}

	public function createAssetCreationHistory(array $data, User $user, int $assetId, int $count)
	{
		$description = $this->getAssetHistoryCreateDescription($data, $assetId);

		if (isset($data['user_id'])) {

			return AssetHistoryModel::create([
				'user_id' => $user->id,
				'asset_id' => $assetId,
				'ticket_no' => request('ticket_no' . $count),
				'action' => 'created',
				'new_user_id' => $data['user_id'],
				'description' => $description,
			]);
		}

		return AssetHistoryModel::create([
			'user_id' => $user->id,
			'asset_id' => $assetId,
			'ticket_no' => request('ticket_no' . $count) ?? ($data['ticket_no'] ?? null),
			'action' => 'created',
			'new_location_id' => $data['location_id'],
			'description' => $description,
		]);
	}

	public function getAssetHistoryCreateDescription(array $data, int $assetId)
	{
		if (isset($data['user_id'])) {
			return __('history.CreatedForUser', [
				'assetname' => $data['serial_no'],
				'assetid' => $assetId,
				'newuser' => optional(User::find($data['user_id']))->userName,
			]);
		}

		return __('history.Created', [
			'assetname' => $data['serial_no'],
			'assetid' => $assetId,
			'newroomname' => optional(Location::find($data['location_id']))->room_name,
		]);
	}

	public function getAsnAssetReceivedDescription($status, $asset, $comments = null, $inputData = null)
	{
		$namespace = "\App\Services\AssetHistoryDescriptions\\";

		$statusClass = 'AsnAssetReceived';


		$className = $namespace . "HistoryFor" . $statusClass;

		$class = new $className;
		return $class->description($status, $inputData, $asset, $comments);
	}

	public function getAsnAssetCreatedDescription($asset, $comments = null)
	{
		$namespace = "\App\Services\AssetHistoryDescriptions\\";

		$statusClass = 'CdwAssetCreated';
		$status = '';
		$data = [];

		$className = $namespace . 'HistoryFor' . $statusClass;

		$class = new $className();

		return $class->description($status, $data, $asset, $comments);
	}

	public function getJiraDescription(array $data, User $user, int $assetId, int $count)
	{
		$jiraDescription = __('jira.Created', [
			'assetname' => $data['serial_no'],
			'assetid' => $assetId,
			'username' => $user->first_name . ' ' . $user->last_name,
			'useremail' => $user->email,
		]);

		return [
			'description' => $jiraDescription,
			'ticketId' => request('ticket_no' . $count) ?? ($data['ticket_no'] ?? null),
			'userId' => $user->id,
		];
	}

	/**
	 * Returns the asset history description for doscovery tools auto reassigned assets.
	 * @param mixed $data
	 *
	 * @return [type]
	 */
	public function getAssetHistoryAutoAssignedDescription($data)
	{
		$oldUserName = optional(optional($data->asset)->user)->first_name . " " . optional(optional($data->asset)->user)->last_name;
		$newUserName = optional($data->user)->first_name . " " . optional($data->user)->last_name;
		return __('history.AutoAssigned', [
			'assetname' => $data->serial_no,
			'assetid'   => $data->asset_id,
			'userId'   => $data->user_id,
			'olduserId' => $data->asset->user_id,
			'olduser' => $oldUserName,
			'newuser'   => $newUserName,
		]);
	}

	/**
	 * Returns the asset history description for discovery tools auto installed assets.
	 *
	 * @param mixed $discoveryTool
	 *
	 * @return string
	 */
	public function getAutoInstalledAssetHistoryDescription($discoveryTool)
	{
		return __('history.AutoInstalled', [
			'assetName'      => $discoveryTool->serial_no,
			'assetId'        => $discoveryTool->asset_id,
			'oldLocationId'  => $discoveryTool->asset?->location_id,
			'newLocationId'  => $discoveryTool->location_id,
			'oldRoomName'    => $discoveryTool->asset?->location?->room_name,
			'newRoomName'    => $discoveryTool->location?->room_name,
		]);
	}
}
