<?php

namespace App\Services;

use App\Models\Asset;
use Carbon\Carbon;
use App\Models\AssetStatus;
use App\Models\Location;
use App\Repositories\CommonFilterRepository;

class AssetData
{
    public function __construct(protected CommonFilterRepository $commonFilterRepository) {}

    public function getAssetData($assets, $start, $data)
    {
        $parentIndex = $start;

        foreach ($assets as $key => $asset) {
            $parentId = $asset->id;

            $parentCount = is_countable($asset->parentAsset) ? count($asset->parentAsset) : 0;
            if ($parentCount != 0) continue;

            $parentIndex++;

            $assetTag = $asset->assetType ? ($asset->assetType->hasRelationToSlug('mobile_assets') ? route('mobile-assets.show', $asset->id) : route('assets.show', $asset->id)) : route('assets.show', $asset->id);

            $nestedData = $this->getNestedData($asset, $parentIndex, '', '', $assetTag);

            $data[] = $nestedData;

            $childCount = is_countable($asset->childrenAsset) ? count($asset->childrenAsset) : 0;
            if ($childCount <= 0) continue;

            $itemClass = '';
            $countChild = 0;
            $sizeofChild = $childCount;

            foreach ($asset->childrenAsset as $child) {
                $countChild++;
                if (($sizeofChild) ==  $countChild) {
                    $itemClass = "last";
                }

                $assetTag = $child->assetType ? ($asset->assetType->hasRelationToSlug('mobile_assets') ? route('mobile-assets.show', $child->id) : route('assets.show', $child->id)) : route('assets.show', $child->id);

                $nestedData = $this->getNestedData($child, $parentIndex . '.' . $countChild, $itemClass, $parentId, $assetTag);

                $data[] = $nestedData;
            }
        }

        return $data;
    }

    public function getServerAssetData($assets, $start, $data)
    {
        $parentIndex = $start;

        foreach ($assets as $key => $asset) {
            $parentId = $asset->id;

            $parentCount = is_countable($asset->parentAsset) ? count($asset->parentAsset) : 0;
            if ($parentCount != 0) continue;

            $parentIndex++;

            $assetTag = route('av-assets.edit', $asset->id);

            $nestedData = $this->getNestedData($asset, $parentIndex, '', '', $assetTag);

            $data[] = $nestedData;

            $childCount = is_countable($asset->childrenAsset) ? count($asset->childrenAsset) : 0;
            if ($childCount <= 0) continue;

            $itemClass = '';
            $countChild = 0;
            $childIndex = 0;
            $sizeofChild = $childCount;

            foreach ($asset->childrenAsset as $child) {
                $countChild++;
                if (($sizeofChild) ==  $countChild) {
                    $itemClass = "last";
                }
                $assetTag = route('av-assets.edit', $child->id);

                $nestedData = $this->getNestedData($child, $parentIndex . '.' . $countChild, $itemClass, $parentId, $assetTag);

                $data[] = $nestedData;

                $childAssetCount = is_countable($child->childrenAsset) ? count($child->childrenAsset) : 0;
                if ($childAssetCount <= 0) continue;

                $itemSubClass = '';
                $countGrandChild = 0;
                $sizeofGrandChild = $childAssetCount;

                foreach ($child->childrenAsset as $grandChild) {
                    $countGrandChild++;
                    if (($sizeofGrandChild) ==  $countGrandChild) {
                        $itemSubClass = 'grandlast';
                    }

                    $assetTag = route('av-assets.edit', $grandChild->id);
                    $nestedData = $this->getNestedData($grandChild, $parentIndex . '.' . $countChild . '.' . $countGrandChild, $itemSubClass, $parentId, $assetTag);

                    $data[] = $nestedData;
                }
            }
        }
        return $data;
    }

    public function getNetworkAssetData($assets, $start, $data)
    {
        $parentIndex = $start;

        foreach ($assets as $key => $asset) {
            $parentId = $asset->id;

            $parentCount = is_countable($asset->parentAsset) ? count($asset->parentAsset) : 0;
            if ($parentCount != 0) continue;

            $parentIndex++;

            $assetTag = route('network-assets.edit', $asset->id);

            $nestedData = $this->getNestedData($asset, $parentIndex, '', '', $assetTag);

            $data[] = $nestedData;

            $childCount = is_countable($asset->childrenAsset) ? count($asset->childrenAsset) : 0;
            if ($childCount <= 0) continue;

            $itemClass = '';
            $countChild = 0;
            $childIndex = 0;
            $sizeofChild = $childCount;

            foreach ($asset->childrenAsset as $child) {
                $countChild++;
                if (($sizeofChild) ==  $countChild) {
                    $itemClass = "last";
                }
                $assetTag = route('network-assets.edit', $child->id);

                $nestedData = $this->getNestedData($child, $parentIndex . '.' . $countChild, $itemClass, $parentId, $assetTag);

                $data[] = $nestedData;

                $childAssetCount = is_countable($child->childrenAsset) ? count($child->childrenAsset) : 0;
                if ($childAssetCount <= 0) continue;

                $itemSubClass = '';
                $countGrandChild = 0;
                $sizeofGrandChild = $childAssetCount;

                foreach ($child->childrenAsset as $grandChild) {
                    $countGrandChild++;
                    if (($sizeofGrandChild) ==  $countGrandChild) {
                        $itemSubClass = 'grandlast';
                    }

                    $assetTag = route('network-assets.edit', $grandChild->id);
                    $nestedData = $this->getNestedData($grandChild, $parentIndex . '.' . $countChild . '.' . $countGrandChild, $itemSubClass, $parentId, $assetTag);

                    $data[] = $nestedData;
                }
            }
        }
        return $data;
    }


    public function getNestedData($asset, $index, $last, $parent, $assetTag)
    {
        $nestedData['dataLast'] = $last;
        $nestedData['dataId'] = $asset->id;
        $nestedData['dataParent'] = $parent;
        $nestedData['id'] = $index;
        $nestedData['type'] = $asset->assetType ? $asset->assetType->name : '';
        $nestedData['asset_tag'] = "<a href=$assetTag>$asset->asset_tag</a>";
        $nestedData['serial_no'] = "<a href=$assetTag>$asset->serial_no</a>";
        $nestedData['hardware_standard'] = $asset->makeAndModel ? $asset->makeAndModel->makeModelName : '';
        $nestedData['technical_spec'] = $asset->technicalSpec ? $asset->technicalSpec->details : '';
        $nestedData['status'] = $asset->assetStatus ? $asset->assetStatus->name : '';
        $nestedData['user_location'] = generateUserLink($asset->user?->id, $asset->user?->user_name);

        if ($asset->location) {
            $nestedData['user_location'] .= (($asset->user) ? ' / ' : '') . generateLocationLink($asset->location?->id, $asset->location?->room_name);
        }
        $nestedData['carrier'] = $asset->carrier ? $asset->carrier->name : '';
        $nestedData['loaner_return_date'] = $asset->loaner_return_date;
        $nestedData['lease_start_date'] = $asset->lease_start_date;
        $nestedData['lease_end_date'] = $asset->lease_end_date;
        $nestedData['age'] = $asset->assetAge;
        $nestedData['modified_date'] = $asset->latestAssetHistory ? $asset->latestAssetHistory->updated_at : '';
        $nestedData['modified_by'] = generateUserLink($asset->latestAssetHistory?->user?->id, $asset->latestAssetHistory?->user?->user_name);

        return $nestedData;
    }

    public function filter()
    {
        $category = isset(request('form')['category']) ? request('form')['category'] : '';
        $assetType = isset(request('form')['asset_type']) ? request('form')['asset_type'] : '';
        $status = isset(request('form')['status']) ? request('form')['status'] : '';
        $carrier = isset(request('form')['carrier']) ? request('form')['carrier'] : '';
        $makeAndModel = isset(request('form')['make_and_model']) ? request('form')['make_and_model'] : '';
        $technicalSpec = isset(request('form')['technical_spec']) ? request('form')['technical_spec'] : '';
        $location = isset(request('form')['location']) ? request('form')['location'] : '';
        $user = isset(request('form')['user']) ? request('form')['user'] : '';
        $assetAge = request('form')['asset_age_value'];
        $ageUnit = request('form')['asset_age_unit'];
        $leaseStartDateFrom = isset(request('form')['lease_start_date_from']) ? request('form')['lease_start_date_from'] : '';
        $leaseStartDateTo = isset(request('form')['lease_start_date_to']) ? request('form')['lease_start_date_to'] : '';
        $leaseEndDateFrom = isset(request('form')['lease_end_date_from']) ? request('form')['lease_end_date_from'] : '';
        $leaseEndDateTo = isset(request('form')['lease_end_date_to']) ? request('form')['lease_end_date_to'] : '';
        $dateDeployedFrom = isset(request('form')['date_deployed_from']) ? request('form')['date_deployed_from'] : '';
        $dateDeployedTo = isset(request('form')['date_deployed_to']) ? request('form')['date_deployed_to'] : '';
        $loanerReturnDateFrom = isset(request('form')['loaner_return_date_from']) ? request('form')['loaner_return_date_from'] : '';
        $loanerReturnDateTo = isset(request('form')['loaner_return_date_to']) ? request('form')['loaner_return_date_to'] : '';
        $lostDateFrom = isset(request('form')['lost_date_from']) ? request('form')['lost_date_from'] : '';
        $lostDateTo = isset(request('form')['lost_date_to']) ? request('form')['lost_date_to'] : '';
        $building = isset(request('form')['building']) ? request('form')['building'] : '';
        $assetTag = isset(request('form')['asset_tag']) ? request('form')['asset_tag'] : '';
        $serialNo = isset(request('form')['serial_no']) ? request('form')['serial_no'] : '';
        $manufacturer = isset(request('form')['manufacturer']) ? request('form')['manufacturer'] : '';


        $assets = Asset::select('assets.*')->with('location', 'user', 'assetType', 'assetType.assetTabs', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'parentAsset', 'carrier', 'latestAssetHistory.user', 'childrenAsset.user', 'childrenAsset.location', 'childrenAsset.assetType', 'childrenAsset.makeAndModel.manufacturer', 'childrenAsset.technicalSpec', 'childrenAsset.assetStatus', 'childrenAsset.carrier', 'childrenAsset.latestAssetHistory.user');


        if ($category) {
            if ($category == 'mobile') {
                $assets = $assets->mobileAsset();
            }
            if ($category == 'server') {
                $assets = $assets->serverAsset();
                $assets = $assets->doesnthave('parentAsset');
            }
            if ($category == 'network') {
                $assets = $assets->networkAsset();
            }
            if ($category == 'avassets') {
                $assets = $assets->avAsset();
            }
            if ($category == 'research') {
                $assets = $assets->researchAsset();
            }
            if ($category == 'it') {
                $assets = $assets->regularAsset();
            }
        }

        $assets = $assets->relationData($assetType, 'assets.asset_type_id');
        $assets = $assets->relationData($status, 'asset_status_id');
        $assets = $assets->relationData($carrier, 'carrier_id');
        $assets = $assets->relationData($makeAndModel, 'assets.make_and_model_id');
        $assets = $assets->relationData($technicalSpec, 'technical_spec_id');
        $assets = $assets->relationData($location, 'location_id');
        $assets = $assets->relationData($building, 'building');
        $assets = $assets->relationData($user, 'user_id');
        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'makeAndModel', 'manufacturer_id', $manufacturer);

        if ($assetAge) {
            if ($ageUnit == 'day') {
                $date = Carbon::now()->subDays($assetAge);
            }

            if ($ageUnit == 'month') {
                $date = Carbon::now()->subMonths($assetAge);
            }

            if ($ageUnit == 'year') {
                $date = Carbon::now()->subYears($assetAge);
            }

            $assets = $assets->where('assets.created_at', '<=', $date);
        }

        if ($leaseStartDateFrom) {
            $assets = $assets->where('lease_start_date', '>=', convert_to_db_date($leaseStartDateFrom));
        }

        if ($leaseStartDateTo) {
            $assets = $assets->where('lease_start_date', '<=', convert_to_db_date($leaseStartDateTo));
        }

        if ($leaseEndDateFrom) {
            $assets = $assets->where('lease_end_date', '>=', convert_to_db_date($leaseEndDateFrom));
        }

        if ($leaseEndDateTo) {
            $assets = $assets->where('lease_end_date', '<=', convert_to_db_date($leaseEndDateTo));
        }

        if ($dateDeployedFrom) {
            $assets = $assets->where('date_deployed', '>=', convert_to_db_date($dateDeployedFrom));
        }

        if ($dateDeployedTo) {
            $assets = $assets->where('date_deployed', '<=', convert_to_db_date($dateDeployedTo));
        }

        if ($loanerReturnDateFrom) {
            $assets = $assets->where('loaner_return_date', '>=', convert_to_db_date($loanerReturnDateFrom));
        }

        if ($loanerReturnDateTo) {
            $assets = $assets->where('loaner_return_date', '<=', convert_to_db_date($loanerReturnDateTo));
        }

        if ($lostDateFrom) {
            $assets = $assets->where('lost_date', '>=', convert_to_db_date($lostDateFrom));
        }

        if ($lostDateTo) {
            $assets = $assets->where('lost_date', '<=', convert_to_db_date($lostDateTo));
        }

        if ($assetTag) {
            $assetTag = $this->getParentTags($assetTag);

            $assets = $assets->relationData($assetTag, 'asset_tag');
        }

        $assets = $assets->relationData($serialNo, 'serial_no');

        $start = request('start');
        $limit = request('length');
        $count = $assets->count();
        if ($limit != -1) {
            $assets = $assets->offset($start)
                ->limit($limit);
        }

        $assets = $this->assetOrder($assets, $category);
        $assets = $assets->get();
        return compact('assets', 'count');
    }

    public function getParentTags($assetTag)
    {
        if (!is_array($assetTag)) {
            $assetTag = [$assetTag];
        }

        $parentTags = Asset::whereIn('id', Asset::whereIn('asset_tag', $assetTag)->pluck('parent_asset_id'))->pluck('asset_tag')->toArray();

        $assetTag = array_filter(array_merge($assetTag, $parentTags));

        return $assetTag;
    }

    public function assetOrder($assets, $category)
    {
        $columns = array_column(request()->input('columns'), 'data');
        $orderColumn = request()->input('order.0.column');
        $order = $columns[$orderColumn];
        $dir   = request()->input('order.0.dir');

        if ($order == 'asset_type_id') {
            $assets = $assets->join('asset_types', 'assets.asset_type_id', '=', 'asset_types.id')->orderBy('asset_types.name', $dir);
        } else if ($order == 'make_and_model_id') {
            $assets = $assets->join('make_and_models', 'assets.make_and_model_id', '=', 'make_and_models.id')
                ->join('manufacturers', 'make_and_models.manufacturer_id', '=', 'manufacturers.id')
                ->orderBy('manufacturers.slug', $dir)
                ->orderBy('make_and_models.slug', $dir);
        } else if ($order == 'technical_spec') {
            $assets = $assets->join('technical_specs', 'assets.technical_spec_id', '=', 'technical_specs.id')->orderBy('technical_specs.details', $dir);
        } else if ($order == 'status') {
            $assets = $assets->join('asset_statuses', 'assets.asset_status_id', '=', 'asset_statuses.id')->orderBy('asset_statuses.name', $dir);
        } else if ($order == 'user_location') {
            $assets = $assets->leftJoin('locations', 'assets.location_id', '=', 'locations.id')->orderBy('locations.room_name', $dir);
        } else if ($order == 'carrier') {
            $assets = $assets->leftJoin('carriers', 'assets.carrier_id', '=', 'carriers.id')->orderBy('carriers.name', $dir);
        } else if ($order == 'created_at') {
            if ($dir == 'asc') {
                $assets = $assets->orderBy('created_at', 'desc');
            } else {
                $assets = $assets->orderBy('created_at', 'asc');
            }
        } else if ($order == 'modified_date') {
            $assets = $assets->leftJoin('asset_histories', 'asset_histories.asset_id', '=', 'assets.id')->orderBy('asset_histories.created_at', $dir);
        } else if ($order == 'modified_by') {
            $assets = $assets->leftJoin('asset_histories', 'asset_histories.asset_id', '=', 'assets.id')
                ->leftJoin('users', 'asset_histories.user_id', '=', 'users.id')
                ->orderBy('users.first_name', $dir);
        } else if ($order == 'serial_no') {
            if ($dir == 'asc') {
                $assets = $assets->orderBy('serial_no', 'desc');
            } else {
                $assets = $assets->orderBy('serial_no', 'asc');
            }
        } else if ($order == 'age') {
            if ($dir == 'asc') {
                $assets = $assets->orderBy('created_at', 'desc');
            } else {
                $assets = $assets->orderBy('created_at', 'asc');
            }
        } else {
            $assets = $assets->orderBy('id', $dir);
        }

        return $assets;
    }

    public function getOrder($name)
    {
        return find_sort_order($name);
    }

    public function checkRepeatValues($csvDatas)
    {
        $errors = [];
        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $count = $data['count_add'];
            if ($data['serial_no']) {
                for ($j = $count; $j < count($csvDatas); $j++) {

                    $nextData = $csvDatas[$j]['assetData'];
                    if ($nextData['serial_no']) {
                        if ($data['serial_no'] == $nextData['serial_no']) {
                            $errors[-$count] = 'Line no ' . ($count + 1) . ' : The serial # : ' . $data['serial_no'] . ' has duplicate values.';
                        }
                        if ($data['asset_tag'] == $nextData['asset_tag']) {
                            $errors[$count] = 'Line no ' . ($count + 1) . ' : The asset Tag # : ' . $data['asset_tag'] . ' has duplicate values.';
                        }
                    }
                }
            }
        }

        return $errors;
    }

    public function checkRepeatNetworkValues($csvDatas)
    {
        $errors = [];
        foreach ($csvDatas as $csvData) {
            $data = $csvData['assetData'];
            $count = $data['count_add'];
            if ($data['serial_no']) {
                for ($j = $count; $j < count($csvDatas); $j++) {

                    $nextData = $csvDatas[$j]['assetData'];
                    if ($nextData['serial_no']) {
                        if ($data['asset_tag'] == $nextData['asset_tag']) {
                            $errors[$count] = 'Line no ' . ($count + 1) . ' : The asset Tag # : ' . $data['asset_tag'] . ' has duplicate values.';
                        }
                        if ($data['serial_no'] == $nextData['serial_no']) {
                            $errors[-$count] = 'Line no ' . ($count + 1) . ' : The serial # : ' . $data['serial_no'] . ' has duplicate values.';
                        }
                        if ((array_key_exists('ip', $data)) && (array_key_exists('ip', $nextData))) {
                            if ($data['ip']) {
                                if ($data['ip'] == $nextData['ip']) {
                                    $errors['ip' . $count] = 'Line no ' . ($count + 1) . ' : The ip address : ' . $data['ip'] . ' has duplicate values.';
                                }
                            }
                        }
                        if ((array_key_exists('mac', $data)) && (array_key_exists('mac', $nextData))) {
                            if ($data['mac']) {
                                if ($data['mac'] == $nextData['mac']) {
                                    $errors['mac' . $count] = 'Line no ' . ($count + 1) . ' : The mac address : ' . $data['mac'] . ' has duplicate values.';
                                }
                            }
                        }
                    }
                }
            }
        }
        return $errors;
    }

    /**
     * Checks if there are any duplicate serial numbers in the upload data.
     *
     * @param array $csvDatas The upload data.
     *
     * @return array An array containing the errors and the valid data.
     */
    public function checkRepeatUploadValues($csvDatas)
    {
        $errors = [];
        $validData = [];

        $csvDatas = collect($csvDatas);

        $duplicates = $csvDatas->groupBy(function ($item) {
            return $item['assetData']['serial_no'];
        })->filter(function ($group) {
            return $group->count() > 1;
        });

        foreach ($duplicates as $serialNo => $group) {
            foreach ($group as $data) {
                $errors[] = 'Line no ' . ($data['assetData']['asset_count'] + 1) . ' : The serial # ' . $serialNo . ' has duplicates.';
            }
        }

        $validData = $csvDatas->filter(function ($data) use ($duplicates) {
            return !$duplicates->has($data['assetData']['serial_no']);
        })->values()->toArray();

        return ['errors' => $errors, 'csvData' => $validData];
    }

    public function checkCanUpdatedStatus($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];
            if (isset($data['asset_status_id'])) {
                $currentStatusId = $data['asset_status_id'];
                $canUpdatedStatuses = $this->searchFromStatus('assigned');
                $currentStatusSlug = AssetStatus::find($data['asset_status_id'])->slug;
                if (!in_array($currentStatusSlug, $canUpdatedStatuses)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The asset with serial # : ' . $data['serial_no'] . ' cannot be updated to assigned.';
                }
            }
        }
        return $errors;
    }

    public function checkCanUpdatedStatusInstall($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];
            if ($data['asset_status_id']) {
                $currentStatusId = $data['asset_status_id'];
                $canUpdatedStatuses = $this->searchFromStatus('installed');
                $currentStatusSlug = AssetStatus::find($data['asset_status_id'])->slug;
                if (!in_array($currentStatusSlug, $canUpdatedStatuses)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The asset with serial # : ' . $data['serial_no'] . ' cannot be updated to installed.';
                }
            }
        }
        return $errors;
    }

    public function searchFromStatus($status)
    {
        $array = config('asset-status');
        $return = array();
        foreach ($array as $key => $val) {
            if (in_array($status, $val)) {
                $return[] = $key;
            }
        }
        return $return;
    }

    public function checkInstallLocation($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];
            if ($data['location_id']) {
                $locationId = $data['location_id'];
                $location = Location::where('id', $locationId)->with('locationType')->first();
                if ($location->locationType->location_type_name !== 'Install') {
                    $errors[$rowCount] = "Line no " . ($rowCount) . " : The Asset can't be installed to install location.";
                }
            }
        }
        return $errors;
    }

    public function checkStorageLocation($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];
            if ($data['location_id']) {
                $locationId = $data['location_id'];
                $location = Location::where('id', $locationId)->with('locationType')->first();
                if ($location->locationType->location_type_name !== 'Storage') {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The Asset can be Returned to Storage location only.';
                }
            }
        }
        return $errors;
    }

    public function checkCanUpdatedStatusLoaned($csvDatas)
    {
        $errors = [];
        $rowCount = 1;
        foreach ($csvDatas as $csvData) {
            $rowCount++;
            $data = $csvData['assetData'];
            if ($data['asset_status_id']) {
                $currentStatusId = $data['asset_status_id'];
                $canUpdatedStatuses = $this->searchFromStatus('loaned');
                $currentStatusSlug = AssetStatus::find($data['asset_status_id'])->slug;
                if (!in_array($currentStatusSlug, $canUpdatedStatuses)) {
                    $errors[$rowCount] = 'Line no ' . ($rowCount) . ' : The asset with serial # : ' . $data['serial_no'] . ' cannot be updated to loaned.';
                }
            }
        }
        return $errors;
    }

    public function getResearchAssetData($assets, $start, $data)
    {
        $parentIndex = $start;

        foreach ($assets as $key => $asset) {
            $parentId = $asset->id;

            $parentCount = is_countable($asset->parentAsset) ? count($asset->parentAsset) : 0;
            if ($parentCount != 0) continue;

            $parentIndex++;

            $assetTag = route('research-assets.edit', $asset->id);

            $nestedData = $this->getNestedData($asset, $parentIndex, '', '', $assetTag);

            $data[] = $nestedData;

            $childCount = is_countable($asset->childrenAsset) ? count($asset->childrenAsset) : 0;
            if ($childCount <= 0) continue;

            $itemClass = '';
            $countChild = 0;
            $childIndex = 0;
            $sizeofChild = count($asset->childrenAsset);

            foreach ($asset->childrenAsset as $child) {
                $countChild++;
                if (($sizeofChild) ==  $countChild) {
                    $itemClass = "last";
                }
                $assetTag = route('research-assets.edit', $child->id);

                $nestedData = $this->getNestedData($child, $parentIndex . '.' . $countChild, $itemClass, $parentId, $assetTag);

                $data[] = $nestedData;

                $childAssetCount = is_countable($child->childrenAsset) ? count($child->childrenAsset) : 0;
                if ($childAssetCount <= 0) continue;

                $itemSubClass = '';
                $countGrandChild = 0;
                $sizeofGrandChild = $childAssetCount;

                foreach ($child->childrenAsset as $grandChild) {
                    $countGrandChild++;
                    if (($sizeofGrandChild) ==  $countGrandChild) {
                        $itemSubClass = 'grandlast';
                    }

                    $assetTag = route('research-assets.edit', $grandChild->id);
                    $nestedData = $this->getNestedData($grandChild, $parentIndex . '.' . $countChild . '.' . $countGrandChild, $itemSubClass, $parentId, $assetTag);

                    $data[] = $nestedData;
                }
            }
        }
        return $data;
    }
}
