<?php

namespace App\Services;

use App\Models\Asset;
use Facades\App\Repositories\BulkUpload;
use Facades\App\Services\Asset\AssetStatusService;
use Illuminate\Support\Facades\Validator;
use App\Models\MakeAndModel;
use App\Models\AssetStatus;
use App\Rules\BulkUploadDateFormat;
use App\Rules\CheckLocationFreezeRule;

class AssetBulkUpload
{
    public function importAssetData($path)
    {
        $count = 1;
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {
            foreach ($items as $key => $item) {
                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData['csvData'][] = $this->generateCSVData($item, $count);
            }
        }
        return $csvData;
    }

    public function getItems($path)
    {
        if (!file_exists($path)) {
            return false;
        }
        $items = csv_to_array($path);
        if (empty($items)) {
            return false;
        }
        return $items;
    }

    public function generateCSVData($item, $count)
    {
        $headerMap = Config('bulk-upload.assetData');
        if (empty($item)) {
            return false;
        }
        $assetData    = [];
        foreach ($headerMap as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);
            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "make_and_model_id") {
                if (!empty($item[$fileHeader])) {
                    $makeModel = MakeAndModel::where('slug', trim(str_slug($item[$fileHeader], '_')))->first();
                    if ($makeModel) {
                        $assetData['asset_type_id'] = $makeModel->asset_type_id;
                        $assetData['manufacturer_id'] = $makeModel->manufacturer_id;
                        $assetData['make_and_model_id'] = $makeModel->id;
                    }
                    if (!$makeModel) {
                        $assetData['asset_type_id'] = null;
                        $assetData['manufacturer_id'] = null;
                        $assetData['make_and_model_id'] = null;
                    }
                }
            }
            if ($dbField == "asset_tag") {
                $assetData['asset_tag'] = $this->getAssetTag($item[$fileHeader]);
            }
        }

        $assetData['count_add'] = $count++;
        session(['count_add' => $assetData['count_add']]);

        return compact('assetData');
    }

    public function importAssetItData($path, $count)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {

            foreach ($items as $key => $item) {

                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData['csvData'][] = $this->generateAssetCSVData($item, $count);
            }
        }
        return $csvData;
    }

    public function importAssetSpecialData($path)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $count = 0;
        if (!empty($items)) {

            foreach ($items as $key => $item) {

                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData = $this->generateAssetSpecialCSVData($item, $count);
                Asset::updateorCreate(['serial_no' => $csvData['assetData']['serial_no']], $csvData['assetData']);
            }
        }
        return true;
    }

    public function generateAssetCSVData($item, $count)
    {

        $headerMap = Config('bulk-upload.assetItData');

        if (empty($item)) {
            return false;
        }

        $assetData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {

            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "make_and_model_id") {
                if (!empty($item[$fileHeader])) {
                    $makeModel = MakeAndModel::where('slug', trim(str_slug($item[$fileHeader], '_')))->first();
                    if ($makeModel) {
                        $assetData['asset_type_id'] = $makeModel->asset_type_id;
                        $assetData['manufacturer_id'] = $makeModel->manufacturer_id;
                        $assetData['make_and_model_id'] = $makeModel->id;
                    }
                    if (!$makeModel) {
                        $assetData['asset_type_id'] = null;
                        $assetData['manufacturer_id'] = null;
                        $assetData['make_and_model_id'] = null;
                    }
                }
            }
            if ($dbField == "asset_tag") {
                $assetData['asset_tag'] = $this->getAssetTag($item[$fileHeader]);
            }
        }

        $assetData['count_add'] = $count++;
        session(['count_add' => $assetData['count_add']]);

        return compact('assetData');
    }

    public function generateAssetSpecialCSVData($item, $count)
    {

        $headerMap = Config('bulk-upload.assetItData');

        if (empty($item)) {
            return false;
        }

        $assetData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {

            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "make_and_model_id") {
                if (!empty($item[$fileHeader])) {
                    $makeModel = MakeAndModel::where('slug', trim(str_slug($item[$fileHeader], '_')))->first();
                    if ($makeModel) {
                        $assetData['asset_type_id'] = $makeModel->asset_type_id;
                        $assetData['make_and_model_id'] = $makeModel->id;
                    }
                    if (!$makeModel) {
                        $assetData['asset_type_id'] = 0;
                        $assetData['make_and_model_id'] = null;
                    }
                }
            }
            $assetData['imei'] = $assetData['serial_no'];
            if ($dbField == "asset_tag") {
                $assetData['asset_tag'] = $this->getAssetTag($item[$fileHeader]);
            }

            $assetStatus = AssetStatusService::whenCreatingAsset();
            $assetStatusId = AssetStatus::where('slug', $assetStatus)->first()->id;
            $assetData['asset_status_id'] = $assetStatusId;
        }

        return compact('assetData');
    }

    public function getRelationalValues($item, $dbField, $fileHeader)
    {
        if ($dbField == "location_id") {
            return BulkUpload::getLocationId($item, $dbField, $fileHeader);
        }

        if ($dbField == "technical_spec_id") {
            return BulkUpload::getTechnicalSpecId($item, $dbField, $fileHeader);
        }

        if ($dbField == "vendor_id") {
            return BulkUpload::getVendorId($item, $dbField, $fileHeader);
        }

        if ($dbField == "carrier_id") {
            return BulkUpload::getCarrierId($item, $dbField, $fileHeader);
        }

        if ($dbField == "parent_asset_id") {
            return BulkUpload::getParentAssetId($item, $dbField, $fileHeader);
        }

        return false;
    }

    public function validateItFields($csvData)
    {
        $count = 2;
        foreach ($csvData as $data) {
            $errors[] = $this->csvItValidator($data['assetData'], $count++);
        }
        return $errors;
    }

    public function csvItValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                // 'ticket_no' => ['required'],
                'po_id' => ['required'],
                'location_id' => ['required', 'exists:locations,id', new CheckLocationFreezeRule($count)],
                'vendor_id' => ['nullable', 'exists:vendors,id'],
                'asset_tag' => ['required', 'unique:assets,asset_tag'],
                'make_and_model_id' => ['required', 'exists:make_and_models,id'],
                'technical_spec_id' => ['required', 'exists:technical_specs,id'],
                'serial_no' => ['required', 'unique:assets,serial_no'],
                'lease_start_date' => ['required_if:asset_type_id,4', new BulkUploadDateFormat($data['lease_start_date'], 'm/d/Y', $count, "lease start")],
                'lease_end_date' => ['required_if:asset_type_id,4', new BulkUploadDateFormat($data['lease_end_date'], 'm/d/Y', $count, "lease end")],
                'warranty_end_date' => [new BulkUploadDateFormat($data['warranty_end_date'], 'm/d/Y', $count, "warranty end")],
            ],
            $messages = [
                'ticket_no.required' => 'Line no ' . $count . ' : The ticket # is required.',
                'po_id.required' => 'Line no ' . $count . ' : The PO # is required.',
                'location_id.required'    => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'    => 'Line no ' . $count . ' : The location does not exist.',
                'vendor_id.exists'    => 'Line no ' . $count . ' : The vendor does not exist.',
                'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                'asset_tag.unique' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'make_and_model_id.required'    => 'Line no ' . $count . ' : The hardware standard is required.',
                'technical_spec_id.required'    => 'Line no ' . $count . ' : The technical specs is required.',
                'make_and_model_id.exists'    => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.exists'    => 'Line no ' . $count . ' : The technical specs does not exist.',
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.unique' => 'Line no ' . $count . ' : The serial # has already been taken.',
                'lease_start_date.required_if' => 'Line no ' . $count . ' : The lease start date is required for printer.',
                'lease_end_date.required_if' => 'Line no ' . $count . ' : The lease end date is required for printer.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    public function validateFields($csvData)
    {
        $count = 2;
        foreach ($csvData as $data) {
            $errors[] = $this->csvValidator($data['assetData'], $count++);
        }
        return $errors;
    }

    public function csvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                'location_id' =>  ['required', 'exists:locations,id', new CheckLocationFreezeRule($count)],
                'asset_type_id' => 'required',
                'asset_tag' => 'required|unique:assets,asset_tag',
                'po_id' => 'required',
                'vendor_id' => ['nullable', 'exists:vendors,id'],
                'serial_no' => 'required|unique:assets,serial_no',
                // 'ticket_no' => 'required',
                'make_and_model_id' => 'required|required_unless:asset_type_id,2',
                'technical_spec_id' => 'required|required_unless:asset_type_id,2',
                'carrier_id' => 'required_if:asset_type_id,6,7',
                'imei' => 'required_if:asset_type_id,6',
                'lease_start_date' => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'lease_end_date' => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
            ],
            $messages = [
                'location_id.required'    => 'Line no ' . $count . ' : The location does not exist.',
                'asset_type_id.required'    => 'Line no ' . $count . ' : The asset type does not exist.',
                'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                'asset_tag.unique' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'po_id.required' => 'Line no ' . $count . ' : The PO # is required.',
                'vendor_id.exists'    => 'Line no ' . $count . ' : The vendor does not exist.',
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.unique' => 'Line no ' . $count . ' : The serial # has already been taken.',
                'ticket_no.required' => 'Line no ' . $count . ' : The ticket # is required.',
                'make_and_model_id.required'    => 'Line no ' . $count . ' : The hardware standard is required.',
                'technical_spec_id.required'    => 'Line no ' . $count . ' : The Technical Spec is required.',
                'make_and_model_id.required_unless'    => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.required_unless'    => 'Line no ' . $count . ' : The technical specs does not exist.',
                'carrier_id.required_if'    => 'Line no ' . $count . ' : The carrier does not exist.',
                'imei.required_if'    => 'Line no ' . $count . ' : IMEI is required.',
                'lease_start_date.required_if' => 'Line no ' . $count . ' : The lease start date is required for Mobile Device.',
                'lease_end_date.required_if' => 'Line no ' . $count . ' : The lease end date is required Mobile Device.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    public function importNetworkAssetData($path, $count)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {

            foreach ($items as $key => $item) {

                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData['csvData'][] = $this->generateNetworkCSVData($item, $count);
            }
        }
        return $csvData;
    }

    public function generateNetworkCSVData($item, $count)
    {
        $headerMap = Config('bulk-upload.networkAssetData');

        if (empty($item)) {
            return false;
        }

        $assetData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {

            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "make_and_model_id") {
                if (!empty($item[$fileHeader])) {
                    $makeModel = MakeAndModel::where('slug', trim(str_slug($item[$fileHeader], '_')))->first();
                    if ($makeModel) {
                        $assetData['asset_type_id'] = $makeModel->asset_type_id;
                        $assetData['manufacturer_id'] = $makeModel->manufacturer_id;
                        $assetData['make_and_model_id'] = $makeModel->id;
                    }
                    if (!$makeModel) {
                        $assetData['asset_type_id'] = null;
                        $assetData['manufacturer_id'] = null;
                        $assetData['make_and_model_id'] = null;
                    }
                }
            }
            if ($dbField == "asset_tag") {
                $assetData['asset_tag'] = $this->getAssetTag($item[$fileHeader]);
            }
        }

        $assetData['count_add'] = $count++;
        session(['count_add' => $assetData['count_add']]);

        return compact('assetData');
    }

    public function validateNetworkFields($csvData)
    {
        $count = 2;
        foreach ($csvData as $data) {
            $errors[] = $this->csvNetworkValidator($data['assetData'], $count++);
        }
        return $errors;
    }

    public function csvNetworkValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                // 'ticket_no' => 'required',
                'po_id' => 'required',
                'location_id' =>  ['required', 'exists:locations,id', new CheckLocationFreezeRule($count)],
                'asset_tag' => 'required|unique:assets,asset_tag',
                'make_and_model_id' => 'required_unless:asset_type_id,2',
                'technical_spec_id' => 'required_unless:asset_type_id,2',
                'serial_no' => 'required|unique:assets,serial_no',
            ],
            $messages = [
                'ticket_no.required' => 'Line no ' . $count . ' : The ticket # is required.',
                'po_id.required' => 'Line no ' . $count . ' : The PO # is required.',
                'location_id.required'    => 'Line no ' . $count . ' : The location does not exist.',
                'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                'asset_tag.unique' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'make_and_model_id.required_unless'    => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.required_unless'    => 'Line no ' . $count . ' : The technical specs does not exist.',
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.unique' => 'Line no ' . $count . ' : The serial # has already been taken.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    public function validateCsvColumns($path, $csvData, $type)
    {
        if ($type == 'it') {
            $headerMap = Config('bulk-upload.assetItData');
        }
        if ($type == 'mobile') {
            $headerMap = Config('bulk-upload.assetData');
        }
        if ($type == 'network') {
            $headerMap = Config('bulk-upload.networkAssetData');
        }
        if ($type == 'it_upload') {
            $headerMap = Config('bulk-upload.uploadAssetItData');
        }
        if ($type == 'mobile_upload') {
            $headerMap = Config('bulk-upload.uploadAssetMobileData');
        }
        if ($type == 'research') {
            $headerMap = Config('bulk-upload.assetResearchData');
        }
        $headerKeys = array_values($headerMap);

        $csv = array_map("str_getcsv", file($path, FILE_SKIP_EMPTY_LINES));
        $csvKeys = array_shift($csv);

        $result = [];

        if (empty($csvData)) {
            return 'Please make sure the file contains data';
        }
        $messages = $this->checkFileDifference($csvKeys, $headerKeys);
        return $messages;
    }

    public function checkFileDifference($csvKeys, $headerKeys)
    {
        if (count($headerKeys) >= count($csvKeys)) {
            $result = array_diff($headerKeys, $csvKeys);
            $messageString = 'Please make sure the column names are spelled correctly. Columns missing - ';
        } else {
            $result = array_diff($csvKeys, $headerKeys);
            $messageString = 'Uploaded file is not match with template file. Extra columns - ';
        }
        $messages = '';

        if (!empty($result)) {
            foreach ($result as $key => $value) {
                $messageString .= $value . ', ';
            }
            $messages = rtrim($messageString, ", ");
        }
        return $messages;
    }

    public function importUploadData($path, $contentData, $count)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {
            foreach ($items as $key => $item) {
                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData['csvData'][] = $this->generateUploadItCSVData($item, $count, $contentData);
            }
        }
        return $csvData;
    }

    public function generateUploadItCSVData($item, $count, $data)
    {
        if ($data['type'] == 'it') {
            $headerMap = Config('bulk-upload.uploadAssetItData');
        } else {
            $headerMap = Config('bulk-upload.uploadAssetMobileData');
        }


        if (empty($item)) {
            return false;
        }

        $assetData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {

            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "asset_tag") {
                $assetData['po_id']             = $data['po_id'];
                $assetData['vendor_id']             = $data['vendor_id'];
                $assetData['location_id']       = $data['location_id'];
                $assetData['ticket_no']         = $data['ticket_no'] ?? null;
                $assetData['asset_type_id']     = $data['asset_type_id'];
                $assetData['make_and_model_id'] = $data['make_and_model_id'];
                $assetData['technical_spec_id'] = $data['technical_spec_id'];
                $assetData['lease_start_date']  = $data['lease_start_date'];
                $assetData['lease_end_date']    = $data['lease_end_date'];
                $assetData['carrier_id']        = $data['carrier_id'];
                $assetData['ip']                = $data['ip'];
                $assetData['mac']               = $data['mac'];
                $assetData['warranty_end_date'] = $data['warranty_end_date'];
                $assetData['asset_tracking_id'] = $data['asset_tracking_id'];
                $assetData['asset_tag'] = $this->getAssetTag($item[$fileHeader]);
                $assetData['asset_original_value'] = $data['asset_original_value'];
            }
        }

        $assetData['count_add'] = $count++;
        session(['count_add' => $assetData['count_add']]);
        return compact('assetData');
    }

    public function validateUploadFields($csvData, $type)
    {
        $count = 2;
        foreach ($csvData as $data) {
            $errors[] = $this->csvUploadValidator($data['assetData'], $type, $count++);
        }
        $valiadationErrors =  empty(array_filter($errors, function ($a) {
            return $a !== null;
        }));
        if (!$valiadationErrors) {
            return $errors;
        }
        return [];
    }

    public function csvUploadValidator($data, $type, $count)
    {
        if ($type == 'avassets') {
            $validator = Validator::make(
                $data,
                [
                    'asset_tag' => 'required',
                    'serial_no' => 'required',
                    'imei' => 'required_if:asset_type_id,6',
                ],
                $messages = [
                    'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                    'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                    'imei.required_if'    => 'Line no ' . $count . ' : IMEI is required.',
                ]
            );
        } else {
            $validator = Validator::make(
                $data,
                [
                    'asset_tag' => 'required|unique:assets,asset_tag',
                    'serial_no' => 'required|unique:assets,serial_no',
                    'imei' => 'required_if:asset_type_id,6',
                ],
                $messages = [
                    'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                    'asset_tag.unique' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                    'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                    'serial_no.unique' => 'Line no ' . $count . ' : The serial # has already been taken.',
                    'imei.required_if'    => 'Line no ' . $count . ' : IMEI is required.',
                ]
            );
        }
        if ($validator->fails()) {
            return $validator->errors();
        }
        return null;
    }

    public function importAvAssetData($path, $count)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {

            foreach ($items as $key => $item) {

                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData['csvData'][] = $this->generateAvCSVData($item, $count);
            }
        }
        return $csvData;
    }

    public function generateAvCSVData($item, $count)
    {
        $headerMap = Config('bulk-upload.avAssetData');

        if (empty($item)) {
            return false;
        }

        $assetData    = [];

        foreach ($headerMap as $dbField => $fileHeader) {

            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "make_and_model_id") {
                if (!empty($item[$fileHeader])) {
                    $makeModel = MakeAndModel::where('slug', trim(str_slug($item[$fileHeader], '_')))->first();
                    if ($makeModel) {
                        $assetData['asset_type_id'] = $makeModel->asset_type_id;
                        $assetData['manufacturer_id'] = $makeModel->manufacturer_id;
                        $assetData['make_and_model_id'] = $makeModel->id;
                    }
                    if (!$makeModel) {
                        $assetData['asset_type_id'] = null;
                        $assetData['manufacturer_id'] = null;
                        $assetData['make_and_model_id'] = null;
                    }
                }
            }
        }

        $assetData['count_add'] = $count++;
        session(['count_add' => $assetData['count_add']]);

        return compact('assetData');
    }

    public function validateAvFields($csvData)
    {
        $count = 2;
        foreach ($csvData as $data) {
            $errors[] = $this->csvAvValidator($data['assetData'], $count++);
        }
        return $errors;
    }

    public function csvAvValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                // 'ticket_no' => 'required',
                'po_id' => 'required',
                'location_id' => 'required|exists:locations,id',
                'asset_tag' => 'required',
                'make_and_model_id' => 'required_unless:asset_type_id,2',
                'technical_spec_id' => 'required_unless:asset_type_id,2',
                'serial_no' => 'required',
            ],
            $messages = [
                'ticket_no.required' => 'Line no ' . $count . ' : The ticket # is required.',
                'po_id.required' => 'Line no ' . $count . ' : The PO # is required.',
                'location_id.required'    => 'Line no ' . $count . ' : The location does not exist.',
                'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                'make_and_model_id.required_unless'    => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.required_unless'    => 'Line no ' . $count . ' : The technical specs does not exist.',
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    public function importAssetResearchData($path)
    {
        $count = 1;
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {
            foreach ($items as $key => $item) {
                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData['csvData'][] = $this->generateResearchCSVData($item, $count);
            }
        }
        return $csvData;
    }

    public function generateResearchCSVData($item, $count)
    {
        $headerMap = Config('bulk-upload.assetResearchData');
        if (empty($item)) {
            return false;
        }
        $assetData    = [];
        foreach ($headerMap as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);
            if ($result !== false) {
                $item[$fileHeader] = $result;
            }
            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "make_and_model_id") {
                if (!empty($item[$fileHeader])) {
                    $makeModel = MakeAndModel::where('slug', trim(str_slug($item[$fileHeader], '_')))->first();
                    if ($makeModel) {
                        $assetData['asset_type_id'] = $makeModel->asset_type_id;
                        $assetData['manufacturer_id'] = $makeModel->manufacturer_id;
                        $assetData['make_and_model_id'] = $makeModel->id;
                    }
                    if (!$makeModel) {
                        $assetData['asset_type_id'] = null;
                        $assetData['manufacturer_id'] = null;
                        $assetData['make_and_model_id'] = null;
                    }
                }
            }
            if ($dbField == "asset_tag") {
                $assetData['asset_tag'] = $this->getAssetTag($item[$fileHeader]);
            }
        }

        $assetData['count_add'] = $count++;
        session(['count_add' => $assetData['count_add']]);

        return compact('assetData');
    }

    public function validateResearchFields($csvData)
    {
        $count = 2;
        foreach ($csvData as $data) {
            $errors[] = $this->csvResearchValidator($data['assetData'], $count++);
        }
        return $errors;
    }

    public function csvResearchValidator($data, $count)
    {
        $validator = Validator::make(
            $data,
            [
                // 'ticket_no' => 'required',
                'po_id' => 'required',
                'location_id' => 'required|exists:locations,id',
                'asset_tag' => 'required|distinct|unique:assets,asset_tag',
                'make_and_model_id' => 'required|exists:make_and_models,id',
                'technical_spec_id' => 'required|exists:technical_specs,id',
                'serial_no' => 'required|unique:assets,serial_no',
            ],
            $messages = [
                'ticket_no.required' => 'Line no ' . $count . ' : The ticket # is required.',
                'po_id.required' => 'Line no ' . $count . ' : The PO # is required.',
                'location_id.required'    => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'    => 'Line no ' . $count . ' : The location does not exist.',
                'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                'asset_tag.distinct' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'asset_tag.unique' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'make_and_model_id.required'    => 'Line no ' . $count . ' : The hardware standard is required.',
                'technical_spec_id.required'    => 'Line no ' . $count . ' : The technical specs is required.',
                'make_and_model_id.exists'    => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.exists'    => 'Line no ' . $count . ' : The technical specs does not exist.',
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.unique' => 'Line no ' . $count . ' : The serial # has already been taken.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    public function getAssetTag($assetTag)
    {
        if (!$assetTag) {
            return null;
        }
        $assetTagLength = strlen($assetTag);
        while ($assetTagLength < 6) {
            $assetTag = '0' . $assetTag;
            $assetTagLength++;
        }
        return $assetTag;
    }
}
