<?php

namespace App\Services\Asset;

use App\Models\Asset;
use App\Models\AssetStatus;
use App\Services\Asset\AssetStatusHistoryService;
use App\Http\Traits\AssetStatusTrait;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;

class UpdateStatusService
{
    use AssetStatusTrait;

    /**
     * Constructor
     * 
     * @param AssetStatusService $assetStatusService
     * @param AssetStatusHistoryService $assetStatusHistoryService
     */
    public function __construct(protected AssetStatusService $assetStatusService, protected AssetStatusHistoryService $assetStatusHistoryService) {}

    /**
     * Update Status of an asset
     * @param array $inputData 
     * param string|null $section
     * @return true or false
     */
    public function updateStatus(array $inputData, $section = null)
    {
        $asset = Asset::with('assetStatus')->where('id', $inputData['asset_id'])->first();

        $data = $this->getData($asset, $inputData);

        $this->assetStatusHistoryService->createAssetHistory($inputData['status_name'], $data, $asset, $inputData, $section);

        $asset->update($data);

        if ($asset->has('childrenAsset')) {
            foreach ($asset->childrenAsset as $assetChild) {
                $this->assetStatusHistoryService->createAssetHistory($inputData['status_name'], $data, $assetChild, $inputData, $section);

                $assetChild->update($data);
            }
        }

        return true;
    }

    /**
     * Fetch the data to update
     * @param Asset $asset 
     * @param array $inputData 
     * @return array
     */
    public function getData(Asset $asset, array $inputData)
    {
        $newStatus = $this->getStatus($inputData['asset_status_id']);

        $assetData['asset_status_id'] = $inputData['asset_status_id'];
        $inputData['status_slug'] = $newStatus?->slug;

        $assetData = $this->arraySetNull($assetData);
        $assetData = $this->arraySetUser($assetData, $newStatus->has_user, $asset, $inputData);
        $assetData = $this->arraySetLocation($assetData, $newStatus->has_location, $asset, $inputData);
        $assetData = $this->arraySetFromRequest($assetData, $inputData);
        $assetData = $this->arraySetEwasteCertficate($assetData, $inputData, $asset);

        return $assetData;
    }

    /**
     * Set Null values for the array keys
     * @param array $assetData 
     * @return array
     */
    public function arraySetNull(array $assetData)
    {
        $assetData['loaner_return_date'] = null;
        $assetData['loaner_retention_date'] = null;
        $assetData['wipe_confirmation'] = null;
        $assetData['lost_date'] = null;
        $assetData['end_of_life_date'] = null;
        $assetData['location_id'] = null;
        $assetData['user_id'] = null;

        return $assetData;
    }

    /**
     * Set user_id value according to the status
     * @param array $assetData 
     * @param int $hasUser 
     * @param Asset $asset 
     * @param array $inputData 
     * @return array
     */
    public function arraySetUser(array $assetData, int $hasUser, Asset $asset, array $inputData)
    {
        if ($hasUser === 1) {
            // If has user, set user_id for some statuses retain the value if not passed from input
            $assetData['user_id'] = $inputData['user_id'] ??  $asset->user_id;
        }

        return $assetData;
    }

    /**
     * Set location_id value according to the status
     * @param array $assetData 
     * @param int $hasLocation 
     * @param Asset $asset 
     * @param array $inputData 
     * @return array
     */
    public function arraySetLocation(array $assetData, int $hasLocation, Asset $asset, array $inputData)
    {
        if ($hasLocation === 1) {
            $assetData['location_id'] = $inputData['location_id'] ?? $asset->location_id;
        }

        return $assetData;
    }

    /**
     * Set array values according to the request
     * @param array $assetData 
     * @param array $inputData 
     * @return array
     */
    public function arraySetFromRequest(array $assetData, array $inputData)
    {
        $assetData['ticket_no'] = $inputData['ticket_no'] ?? null;
        $assetData['wipe_confirmation'] = $inputData['wipe_confirmation'] ?? null;
        $assetData['retain_hold_wipe_confirmed_date'] = $inputData['retain_hold_wipe_confirmed_date'] ?? null;
        $assetData['retain_hold_note'] = $inputData['retain_hold_note'] ?? null;

        $this->handleLoanedStatus($inputData, $assetData);
        $this->handleLostOrStolenStatus($inputData, $assetData);
        $this->handleRetainHoldStatus($inputData, $assetData);
        $this->handleEndOfLifeStatus($inputData, $assetData);
        $this->handleDisposedOrWipedStatus($inputData, $assetData);

        return $assetData;
    }

    /**
     * Set the loaner_return_date value according to the loaned status
     * If no date is provided, set it to the current date plus the configured loaner period
     * @param array $inputData 
     * @param array $assetData 
     * @return void
     */
    private function handleLoanedStatus(array $inputData, array &$assetData)
    {
        if ($inputData['status_slug'] == 'loaned') {
            $date = $inputData['loaner_return_date'];

            if (empty($date)) {
                $date = parse_date_from_db_datetime(Carbon::now()->addDays(config('date.period.loaner')));
            }

            $assetData['loaner_return_date'] = $date;
        }
    }

    /**
     * Handle lost or stolen status
     * If the status is lost or stolen, set the lost_date value
     * @param array $inputData 
     * @param array $assetData 
     * @return void
     */
    private function handleLostOrStolenStatus(array $inputData, array &$assetData)
    {
        if (in_array($inputData['status_slug'], ['lost_or_stolen', 'stolen_lost'])) {
            $assetData['lost_date'] = $inputData['lost_date'];
        }
    }

    /**
     * Set the loaner_retention_date value according to the retain hold status
     * If the status is retain hold, set the loaner_retention_date value
     * @param array $inputData 
     * @param array $assetData 
     * @return void
     */
    private function handleRetainHoldStatus(array $inputData, array &$assetData)
    {
        if ($inputData['status_slug'] == 'retain_hold') {
            $assetData['loaner_retention_date'] = $inputData['loaner_retention_date'];
        }
    }

    /**
     * Set the end_of_life_date value according to the end of life status
     * If the status is end of life, set the end_of_life_date value
     * @param array $inputData 
     * @param array $assetData 
     * @return void
     */
    private function handleEndOfLifeStatus(array $inputData, array &$assetData)
    {
        if ($inputData['status_slug'] == 'end_of_life') {
            $assetData['end_of_life_date'] = $inputData['end_of_life_date'] ?? date('m/d/Y');
        }
    }

    /**
     * Handle disposed or wiped status.
     * If the status is end_of_life_disposed, end_of_life_data_wiped, or rma_returned,
     * set the vendor_id value in the asset data.
     *
     * @param array $inputData The input data containing the status and vendor information.
     * @param array &$assetData The asset data to be updated with the vendor_id.
     * @return void
     */

    private function handleDisposedOrWipedStatus(array $inputData, array &$assetData)
    {
        if (in_array($inputData['status_slug'], ['end_of_life_data_wiped', 'rma_returned'])) {
            $assetData['vendor_id'] = $inputData['vendor_id'] ?? null;
        }
    }

    /**
     * Store ewaste certificate
     * @param array $assetData 
     * @return array
     */
    public function arraySetEwasteCertficate(array $assetData, array $inputData, Asset $asset)
    {
        if (array_key_exists('ewaste_certificate', $inputData)) {
            $fileName = $asset->serial_no . '_' . date('d_m_Y_H_i_s') . ".pdf";
            $path  = 'uploads/eol_certificates/' . date('Y_m_d');
            $assetData['ewaste_certificate'] = $inputData['ewaste_certificate']->storeAs($path, $fileName, 'public');
            $assetData['certificate_added_date'] = Carbon::now()->format('m/d/Y');
        }

        return $assetData;
    }

    /**
     * Updates the status of assets belonging to a terminated user to pending return.
     * This is a utility function used by the UserObserver to update the status of all assets
     * belonging to a user when the user is terminated.
     * 
     * @param User $user The user whose assets need to be updated
     * @return bool
     */
	public function updateTerminatedUserAssetsToPendingReturn($user)
	{
        $pendingReturnStatus = Cache::remember('pending-return-asset-status', 7200, function () {
            return AssetStatus::where('slug', 'pending_return')->first();
        });

        $includedStatuses = Cache::remember('pending-return-included-asset-statuses', 7200, function () {
            return AssetStatus::whereIn('slug', ['assigned', 'loaned'])->pluck('id')->toArray();
        });

		$userAssets = Asset::with('assetStatus', 'user')
            ->where('user_id', $user->id)
            ->whereIn('asset_status_id', $includedStatuses)
            ->get();

        foreach ($userAssets as $asset) {
            $this->assetStatusHistoryService->createPendingReturnAutoUpdateHistory($asset, $pendingReturnStatus);

            $asset->update([
                'asset_status_id' => $pendingReturnStatus->id,
                'ticket_no' => null
            ]);
        }

        return true;
	}
}
