<?php

namespace App\Services\Asset;

use App\Models\Asset;
use App\Models\Location;
use App\Models\UserType;
use App\Repositories\AssetRepository;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SearchAssetService
{
    public function __construct(
        protected AssetRepository $assetRepository
    ) {}

    /**
     * Retrieves search fields based on the provided search and type parameters.
     *
     * @param mixed $search The search criteria.
     * @param mixed $type The type of search.
     * @return \Illuminate\Support\Collection The merged collection of search fields.
     */
    public function getSearchFields($search, $type)
    {
        $resultAssetTag = $this->getSearchAssetTagFields($search, $type);
        $resultSerial   = $this->getSearchSerialFields($search, $type);
        $resultUser     = $this->getSearchUserFields($search);
        $resultLocation = $this->getSearchLocationFields($search);

        // Put all collections into an array
        $collections = collect([
            $resultAssetTag,
            $resultSerial,
            $resultUser,
            $resultLocation,
        ]);

        // Filter out empty collections and merge the remaining ones
        $result = $collections
            ->filter(function ($collection) {
                return $collection->isNotEmpty();
            })
            ->reduce(function ($carry, $collection) {
                return $carry->merge($collection);
            }, collect())->unique();

        return $this->prepareOutput($result, $type);
    }

    /**
     * Provides data to populate the autocomnplte input field with Asset and Serial#
     * @param mixed $search
     * @param mixed $type
     *
     * @return [type]
     */
    public function getSearchSerialAndAssetTags($search, $type)
    {
        $resultAssetTag = $this->getSearchAssetTagFields($search, $type);
        $resultSerial   = $this->getSearchSerialFields($search, $type);

        $result = $resultAssetTag->union($resultSerial);

        return $this->prepareOutput($result, $type);
    }

    /**
     * Retrieves a list of filtered users with specific fields based on the given search term.
     *
     * @param string $search The search term to filter the users.
     *
     * @return array Returns an array of users with their corresponding IDs and concatenated fields.
     */
    public function getSearchUserFields($search)
    {
        $res = User::superAdmin()
            ->where(function ($query) use ($search) {
                $query->where('first_name', 'like', $search . '%')
                    ->orWhere('last_name', 'like', $search . '%')
                    ->orWhereRaw("CONCAT(`first_name`, ' ', `last_name`) like ?", [$search . '%'])
                    ->orWhere('email', 'like', $search . '%');
            })
            ->limit(15)->get();

        $result = $res->mapWithKeys(function ($item, $key) {
            return [
                $key => [
                    'value' => $item->email,
                    'label' => $item->first_name . ' ' . $item->last_name . ' - ' . $item->email,
                ]
            ];
        });

        return $result;
    }

    /**
     * Retrieves locations to populate the autocomplete input field based on the given
     * search criteria.
     *
     * @param string $search The search criteria.
     *
     * @return \Illuminate\Support\Collection The collection of search fields.
     */
    public function getSearchLocationFields($search)
    {
        $res = Location::where('room_name', 'like', '%' . $search . '%')->limit(15)->get();

        return $this->formatResults($res, 'room_name');
    }

    /**
     * Gets the assets to populate the autocomplete input field based on the given
     * search criteria, type and field.
     *
     * @param string $search The search criteria.
     * @param string $type The type of search.
     * @param string $field The field to search in.
     *
     * @return \Illuminate\Support\Collection The collection of search fields.
     */
    public function getSearchAssetFields($search, $type, $field)
    {
        $res = Asset::where($field, 'like', '%' . $search . '%')->with('assetType.assetTabs');
        $res = $this->getAssetFromTypes($res, $type);
        $res = $res->limit(15)->get();

        return $this->formatResults($res, $field);
    }

    /**
     * Formats the given results into a collection suitable for being returned as JSON
     * for an autocomplete input field.
     *
     * @param \Illuminate\Support\Collection $res The collection of results.
     * @param string|null $field The field to use as the value and label for the item.
     *
     * @return \Illuminate\Support\Collection The formatted collection of results.
     */
    private function formatResults($res, $field = null)
    {
        return $res->mapWithKeys(function ($item, $key) use ($field) {
            return [
                $key => [
                    'value' => $item[$field],
                    'label' => $item[$field],
                    'asset_type_tab' => !empty(optional($item->assetType)->assetTabs) ? optional(optional($item->assetType)->assetTabs[0])->slug : ''
                ]
            ];
        });
    }

    /**
     * Gets the asset tags to populate the autocomplete input field
     * @param string $search The search criteria.
     * @param string $type The type of search.
     *
     * @return \Illuminate\Support\Collection The collection of search fields.
     */
    public function getSearchAssetTagFields($search, $type)
    {
        return $this->getSearchAssetFields($search, $type, 'asset_tag');
    }

    /**
     * Gets the serial numbers to populate the autocomplete input field.
     *
     * @param string $search The search criteria.
     * @param string $type The type of search.
     * @return \Illuminate\Support\Collection The collection of serial number fields.
     */
    public function getSearchSerialFields($search, $type)
    {
        return $this->getSearchAssetFields($search, $type, 'serial_no');
    }

    /**
     * Get assets filtered by type.
     *
     * @param \Illuminate\Database\Eloquent\Builder $assets The query builder for assets.
     * @param string $assetType The type of asset to filter by.
     * @return \Illuminate\Database\Eloquent\Builder The filtered query builder.
     */
    public function getAssetFromTypes(\Illuminate\Database\Eloquent\Builder $assets, string $assetType): \Illuminate\Database\Eloquent\Builder
    {
        return match ($assetType) {
            'general' => $assets,
            'mobile' => $assets->mobileAsset(),
            'network' => $assets->networkAsset(),
            'av' => $assets->avAsset(),
            'research' => $assets->researchAsset(),
            default => $assets->regularAsset(),
        };
    }

    /**
     * Prepare the output of the search query by adding a message
     * indicating where to search for the asset if the result is empty.
     *
     * @param \Illuminate\Support\Collection $result The collection of search results.
     * @param string $type The type of search.
     *
     * @return \Illuminate\Support\Collection The prepared collection of search results.
     */
    private function prepareOutput($result, $type)
    {
        if (!$result->count()) {
            $result[] = match ($type) {
                'mobile' => ['value' => '0', 'label' => 'Search for this asset in the "IT Assets Tab or Network Assets Tab or AV Assets Tab"', 'data' => ''],
                'network' => ['value' => '0', 'label' => 'Search for this asset in the "Mobile Assets Tab or IT Assets Tab or AV Assets Tab"', 'data' => ''],
                'av' => ['value' => '0', 'label' => 'Search for this asset in the "Mobile Assets Tab or IT Assets Tab or Network Assets Tab"', 'data' => ''],
                default => ['value' => '0', 'label' => 'Search for this asset in the "Mobile Assets Tab or Network Assets Tab or AV Assets Tab"', 'data' => ''],
            };
        }

        return $result;
    }

    /**
     * Returns a collection of assets that match the given search criteria.
     *
     * @param  string  $searchText  The search text to query the assets with.
     * @param  string  $type  The type of assets to search through.
     * @return array  A collection of assets matching the search criteria, grouped
     * by asset type.
     */
    public function getSearchedAssets($searchText = NULL, $type = 'assets')
    {
        $results = [
            'user' => null,
            'softwareUser' => null,
            'location' => null,
            'searchedLocationAsset' => null,
            'searchedUserAsset' => null,
            'searchedLocationAssetTagCount' => 0,
            'searchedUserAssetTagCount' => 0,
        ];

        if (empty($searchText) || is_null($searchText)) {
            return $results;
        }

        $results['user'] = $this->assetRepository->searchAssetUser($searchText, $type);
        $results['softwareUser'] = $this->assetRepository->searchSoftwareAssetUser($searchText);
        $results['location'] = $this->assetRepository->searchAssetLocation($searchText, $type);

        $results['searchedLocationAsset'] = $this->getSearchedAsset('location', $searchText, $type);
        $results['searchedUserAsset'] = $this->getSearchedAsset('user', $searchText, $type);

        $results['searchedLocationAssetTagCount'] = $results['searchedLocationAsset'] ? 1 : 0;
        $results['searchedUserAssetTagCount'] = $results['searchedUserAsset'] ? 1 : 0;

        $results['location'] = $this->processLocationAssets(
            $results['location'],
            $searchText,
            $results['searchedLocationAsset'],
            $type
        );

        $results['user'] = $this->processUserAssets(
            $results['user'],
            $searchText,
            $results['searchedUserAsset'],
            $type
        );

        if ($results['softwareUser']) {
            $results['softwareUser']->softWareAssets = $results['softwareUser']->userLicense()
                ->with(['license', 'licenseKey'])
                ->paginate(50, ['*'], 'softwarePage');
        }

        return $results;
    }

    /**
     * Returns the first searched asset based on the given search method and text.
     *
     * @param string $searchType The method to use for searching the asset (e.g., 'searchAssetLocation', 'searchAssetUser').
     * @param string $searchText The text to search for in the asset.
     * @param string $type The type of assets to search through.
     *
     * @return \App\Models\AssetModel|null The first matched asset with related data, or null if no asset is found.
     */
    private function getSearchedAsset($searchType, $searchText, $type = 'assets')
    {
        $query = match ($type) {
            'mobile_phone' => Asset::mobileAsset(),
            default => Asset::regularAsset(),
        };

        switch ($searchType) {
            case 'location':
                return $this->assetRepository->assetLocationSearched($query, $searchText)
                    ->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')
                    ->first();
            case 'user':
                return $this->assetRepository->assetUserSearched($query, $searchText)
                    ->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')
                    ->first();
            default:
                return null;
        }
    }

    /**
     * Processes the given location assets by removing the searched asset from the
     * location's assets and paginating the rest. If the searched asset is found,
     * it is prepended to the first page of the location's assets.
     *
     * @param \App\Models\Location $location
     * @param string $searchText
     * @param \App\Models\AssetModel|null $searchedLocationAsset
     * @param string $type
     * @return \App\Models\Location
     */
    private function processLocationAssets($location, $searchText, $searchedLocationAsset, $type = 'assets')
    {
        if (!$location) {
            return $location;
        }

        $location->assets = $location->assets()
            ->with('user:id,first_name,last_name,email,status', 'location:id,room_name', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset:id,asset_tag,serial_no', 'childrenAsset:id,asset_tag,serial_no');

        $location->assets = match ($type) {
            'mobile_phone' => $location->assets->mobileAsset()->notSearched($searchText),
            default => $location->assets->regularAsset()->notSearched($searchText),
        };

        $location->assets = $location->assets->paginate(50);

        if ($location->assets->currentPage() === 1 && $searchedLocationAsset) {
            $location->assets->prepend($searchedLocationAsset);
        }

        return $location;
    }

    /**
     * Processes the given user assets by removing the searched asset from the
     * user's assets and paginating the rest. If the searched asset is found,
     * it is prepended to the first page of the user's assets.
     *
     * @param \App\Models\User $user
     * @param string $searchText
     * @param \App\Models\AssetModel|null $searchedUserAsset
     * @param string $type
     * @return \App\Models\User
     */
    private function processUserAssets($user, $searchText, $searchedUserAsset, $type = 'assets')
    {
        if (!$user) {
            return $user;
        }

        $user->assets = $user->assets()
            ->with('user:id,first_name,last_name,email,status', 'location:id,room_name', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset:id,asset_tag,serial_no', 'childrenAsset:id,asset_tag,serial_no');

        $user->assets = match ($type) {
            'mobile_phone' => $user->assets->mobileAsset()->notSearched($searchText),
            default => $user->assets->regularAsset()->notSearched($searchText),
        };

        $user->assets = $user->assets->paginate(50);

        if ($user->assets->currentPage() === 1 && $searchedUserAsset) {
            $user->assets->prepend($searchedUserAsset);
        }

        return $user;
    }

    /**
     * Determine if the current user is a super user or super admin
     * 
     * @return int 1 if the user is a super user or super admin, 0 otherwise
     */
    public function determineAdminStatus()
    {
        $userTypes = ['Super User', 'Super Admin'];
        $userTypeName = UserType::find(User::find(Auth::id())->user_type_id)->name;

        return in_array($userTypeName, $userTypes) ? 1 : 0;
    }

    /**
     * Get the total original value of assets based on search criteria.
     * 
     * @param string $searchText The text to search for in the asset.
     * @param string $type The type of assets to search through.
     * @param \App\Models\User $user The user to search for.
     * @param array $result The response to return.
     * 
     * @return \Illuminate\Http\JsonResponse The total original value of the assets.
     */
    public function getOriginalValueTotal($searchText, $type, $user, $result)
    {
        if (empty($searchText) || is_null($searchText)) {
            return response()->json($result);
        }

        $user = match ($type) {
            'mobile' => $this->assetRepository->searchAssetUser($searchText, 'mobile_phone'),
            'av' => $this->assetRepository->searchAssetUser($searchText, 'av'),
            'network' => $this->assetRepository->searchAssetUser($searchText, 'network'),
            'research' => $this->assetRepository->searchAssetUser($searchText, 'research'),
            default => $this->assetRepository->searchAssetUser($searchText),
        };

        if ($user) {
            $user->assets = match ($type) {
                'mobile' => $user->assets()->with('technicalSpec')->mobileAsset(),
                'av' => $user->assets()->with('technicalSpec')->avAsset(),
                'network' => $user->assets()->with('technicalSpec')->networkAsset(),
                'research' => $user->assets()->with('technicalSpec')->researchAsset(),
                default => $user->assets()->with('technicalSpec')->regularAsset(),
            };

            $userSum = $this->assetRepository->sumOriginal($user->assets->pluck('id'));
            $result = ['sum' => number_format((float)$userSum)];
        }

        return response()->json($result);
    }

    /**
     * Returns the total original value of assets in a location based on search criteria.
     * 
     * @param string $searchText The text to search for in the asset.
     * @param string $type The type of assets to search through.
     * @param \App\Models\Location $location The location to search in.
     * @param array $result The response to return.
     * 
     * @return \Illuminate\Http\JsonResponse The total original value of the assets in the location.
     */
    public function getOriginalValueLocationTotal($searchText, $type, $location, $result)
    {
        if (empty($searchText) || is_null($searchText)) {
            return response()->json($result);
        }

        $location = match ($type) {
            'mobile' => $this->assetRepository->searchAssetLocation($searchText, 'mobile_phone'),
            'av' => $this->assetRepository->searchAssetLocation($searchText, 'av'),
            'network' => $this->assetRepository->searchAssetLocation($searchText, 'network'),
            'research' => $this->assetRepository->searchAssetLocation($searchText, 'research'),
            default => $this->assetRepository->searchAssetLocation($searchText),
        };

        if ($location) {
            $location->assets = match ($type) {
                'mobile' => $location->assets()->with('technicalSpec')->mobileAsset(),
                'av' => $location->assets()->with('technicalSpec')->avAsset(),
                'network' => $location->assets()->with('technicalSpec')->networkAsset(),
                'research' => $location->assets()->with('technicalSpec')->researchAsset(),
                default => $location->assets()->with('technicalSpec')->regularAsset(),
            };

            $locationSum = $this->assetRepository->sumOriginal($location->assets->pluck('id'));
            $result = ['sum' => number_format(floatval($locationSum))];
        }

        return response()->json($result);
    }
}
