<?php

namespace App\Services\Asset;

use App\Events\BulkUpdates;
use App\Models\Asset;
use App\Services\Integrations\Tickets\TicketManagementService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class LinkAssetService
{
    /**
     * Initializes the class with a TicketManagementService instance.
     *
     * @param TicketManagementService $ticketManagementService The service for managing tickets.
     */
    public function __construct(protected TicketManagementService $ticketManagementService) {}

    /**
     * Links a child asset to a parent asset, creating ticket and asset history records.
     *
     * Validates the input data before proceeding with the linking operation. 
     * If the validation fails, the function returns false.
     *
     * @param array $inputData The input data containing asset serial numbers and ticket information.
     * @param User $user The user performing the linking operation.
     * @param string $i Optional index for handling multiple assets in bulk operations.
     *
     * @return bool Returns true if the assets are linked successfully, false if validation fails.
     */
    public function linkAsset($inputData, $user, $i = '')
    {
        if ($this->validateLink($inputData, $i) === true) {
            return false;
        }

        $childAsset = Asset::getBySerial($inputData['child_serial_no' . $i])->first();
        $parentAsset = Asset::getBySerial($inputData['parent_serial_no' . $i])->first();

        $this->createTicketHistory($childAsset, $parentAsset, $user, $inputData, $i);

        $this->createAssetHistory($childAsset, $parentAsset, $inputData, $i);

        $this->updateLink($childAsset, $parentAsset);

        return true;
    }

    /**
     * Validates the link operation by checking if the input data contains the required fields.
     *
     * @param array $inputData The input data to be validated.
     * @param string $i The index of the asset being linked.
     *
     * @return bool True if the validation fails, false otherwise.
     */
    public function validateLink($inputData, $i = '')
    {
        $childSerialNoKey = 'child_serial_no' . $i;
        $parentSerialNoKey = 'parent_serial_no' . $i;
        $ticketNoKey = 'ticket_no' . $i;

        $validator = Validator::make($inputData, [
            $childSerialNoKey => ['required',  'regex:/^[A-Z0-9\-_)(]{2,20}$/'],
            $parentSerialNoKey => ['required',  'regex:/^[A-Z0-9\-_)(]{2,20}$/'],
            // $ticketNoKey => ['required', 'string'],
        ]);

        return $validator->fails();
    }


    /**
     * Creates a ticket history for the given asset and parent asset.
     *
     * @param mixed $asset The asset to create the ticket history for.
     * @param mixed $parentAsset The parent asset to create the ticket history for.
     * @param mixed $user The user creating the ticket history.
     * @param mixed $inputData The input data containing ticket information.
     * @param mixed $i The index of the ticket information in the input data.
     *
     * @return bool True if the ticket history was created successfully.
     */
    public function createTicketHistory($asset, $parentAsset, $user, $inputData, $i = null)
    {
        $ticketCommentstext =  formatTicketComment($inputData['comments' . $i]);
        $ticketDescription = __('jira.Linked', [
            'parentAssetId' => $parentAsset->id,
            'childAssetId' => $asset->id,
            'parentAssetname' => $inputData['parent_serial_no' . $i],
            'childAssetname' => $inputData['child_serial_no' . $i],
            'commenttext' => $ticketCommentstext,
            'username' => $user->first_name . ' ' . $user->last_name,
            'useremail' => $user->email,
        ]);

        $this->ticketManagementService->addComment(Auth::id(), ($inputData['ticket_no' . $i] ?? null), $ticketDescription, "Asset Linked");

        return true;
    }


    /**
     * Creates a new asset history entry for the given asset and parent asset.
     *
     * @param mixed $asset The asset to create the history entry for.
     * @param mixed $parentAsset The parent asset to create the history entry for.
     * @param mixed $inputData The input data containing asset information.
     * @param mixed $i The index of the asset information in the input data.
     *
     * @return bool True if the asset history entry was created successfully.
     */
    public function createAssetHistory($asset, $parentAsset, $inputData, $i = null)
    {
        $commentstext = ($inputData['comments' . $i] != NULL ?  " with comments \"" . $inputData['comments' . $i] . "\"" : "");
        $description = __('history.Linked', [
            'parentAssetId' => $parentAsset->id,
            'childAssetId' => $asset->id,
            'parentAssetname' => $inputData['parent_serial_no' . $i],
            'childAssetname' => $inputData['child_serial_no' . $i],
            'commenttext' => $commentstext,
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'asset_id' => $asset->id,
            'ticket_no' => $inputData['ticket_no' . $i] ?? null,
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'linked',
            'comments' => $inputData['comments' . $i],
            'old_parent_asset_id' => $asset->parent_asset_id,
            'new_parent_asset_id' => $parentAsset->id,
            'old_location_id' => $asset->location_id,
            'new_location_id' => $parentAsset->location_id,
            'old_user_id' => $asset->user_id,
            'new_user_id' => $parentAsset->user_id,
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $parentAsset->asset_status_id,
            'old_value' => $inputData['parent_serial_no' . $i],
            'new_value' => $inputData['child_serial_no' . $i],
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return true;
    }

    /** Links the asset
     * @param mixed $asset
     * @param mixed $parentAsset
     * 
     * @return [type]
     */
    public function updateLink($asset, $parentAsset)
    {

        return $asset->update([
            'parent_asset_id' => $parentAsset->id,
            'user_id' => $parentAsset->user_id,
            'location_id' => $parentAsset->location_id,
            'asset_status_id' => $parentAsset->asset_status_id,
            'carrier_id' => $parentAsset->carrier_id,
            'linked_date' => Carbon::now()
        ]);
    }

    /**
     * Removes the links between the given assets and their parents.
     * 
     * @param array $links An array of asset IDs to be unlinked.
     * 
     * @return bool True if the links were removed successfully, false otherwise.
     */
    public function removeLinks($links)
    {
        try {
            foreach ($links as $link) {
                $asset = Asset::where('id', $link)->first();

                if ($asset->parent_asset_id == null) {
                    continue;
                }

                $description = "";

                $parent =  Asset::where('id', $asset->parent_asset_id)->first();
                $description = __('history.Unlinked', [
                    'assetid' => $asset->id,
                    'parent_assetid' => $parent->id,
                    'assetname' => $asset->serial_no,
                    'parentAssetname' => $parent->serial_no,
                    'oldroomname' => $parent
                ]);

                $assetHistory = [
                    'user_id' => Auth::id(),
                    'asset_id' => $asset->id,
                    'ticket_no' => '',
                    'ticket_service_provider' => config('ticket-integration.service'),
                    'action' => 'unlink',
                    'comments' => '',
                    'old_parent_asset_id' => $asset->parent_asset_id,
                    'new_parent_asset_id' => null,
                    'old_value' => $parent->serial_no,
                    'new_value' => null,
                    'description' => $description,
                    'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                    'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                ];

                event(new BulkUpdates($assetHistory));

                Asset::where('id', $link)->update(['parent_asset_id' => null]);
            }

            return true;
        } catch (\Exception) {
            return false;
        }
    }
}
