<?php

namespace App\Services\Asset;

use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\UserType;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class AssetStatusService
{
	/**
	 * The status that should be set to a newly created asset
	 *
	 * @return string The status slug
	 */
	public function whenCreatingAsset()
	{
		return 'brand_new'; //status that should be set to a newly created asset
	}

	/**
	 * Get the status for a newly created asset.
	 *
	 * @param Asset $asset
	 * @return Status
	 */
	public function getStatusForNewAsset()
	{
		return AssetStatus::where('slug', $this->whenCreatingAsset())->first();
	}

	/**
	 * Retrieves the class name for a status update based on the provided slug.
	 *
	 * @param string $slug The slug used to generate the status class name
	 * @return string The class name for the status update
	 */
	private function getStatusUpdateClassName($slug)
	{
		$status = Str::ucfirst($slug);
		$statusClass = str_replace(' ', '', $status);
		$statusClass = str_replace('/', '', $statusClass);
		$statusClass = str_replace('-', '', $statusClass);
		$slug = str_slug($slug, '_');
		$namespace = "\App\Services\StatusUpdateConditions\\";
		$className = $namespace . "Status$statusClass"; //eg. StatusNew

		if (!class_exists($className)) {
			$className = $namespace . "StatusCommon";
		}

		return $className;
	}

	/**
	 * Check if an asset can be updated to a given status.
	 *
	 * The conditions for the update are checked in the following order:
	 * 1. Whether the asset can be updated from the current status.
	 * 2. Whether the asset can be updated to the given status.
	 * 3. Whether the current status is allowed to be updated to the given status.
	 *
	 * @param string $statusSlug The slug of the status to be updated to.
	 * @param Asset $asset The asset model instance.
	 * @return boolean Whether the asset can be updated to the given status.
	 */
	public function canBeUpdatedTo($statusSlug, $asset)
	{
		$currentStatusClassName = $this->getStatusUpdateClassName(optional($asset->first()->assetStatus)->name);
		$currentStatusClass = new $currentStatusClassName;

		// Checks the conditions for updating from the current status
		if (!$currentStatusClass->canbeUpdatedFromThisStatus($asset)) {
			return false;
		}

		$toStatusClassName = $this->getStatusUpdateClassName($statusSlug);

		$toStatusClass = new $toStatusClassName;
		if (!$toStatusClass->allowedForAsset($asset)) {
			return false;
		}

		$asset = $asset->first();
		if ($asset->asset_status_id) {
			$currentStatusSlug = AssetStatus::find($asset->asset_status_id)->slug;
			$statusSlug = str_slug($statusSlug, '_');
			if (!in_array($statusSlug, config('asset-status.' . $currentStatusSlug))) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Get the available statuses for updating an asset.
	 *
	 * The available statuses are filtered based on the following conditions:
	 * 1. The user type of the current user.
	 * 2. The general type of the asset.
	 *
	 * @param string $statusSlug The slug of the current status of the asset.
	 * @return \Illuminate\Database\Eloquent\Collection The collection of available statuses.
	 */
	public function availableStatusesForUpdate($statusSlug)
	{
		$allowedStatusForAsset = config('asset-status.' . $statusSlug);
		// if ($statusSlug == 'legal_hold') {
		// 	$userTypeId = User::find(Auth::id())->user_type_id;
		// 	$userTypes = ['Administrator', 'Super Admin'];
		// 	$userTypeName = UserType::find($userTypeId)->name;
		// 	if (!in_array($userTypeName, $userTypes)) {
		// 		$allowedStatusForAsset = [];
		// 	}
		// }

		if (request('general_type') == 'av' || request('general_type') == 'network') {
			if (($key = array_search('assigned', $allowedStatusForAsset)) !== false) {
				unset($allowedStatusForAsset[$key]);
			}
		}

		if (!is_array($allowedStatusForAsset)) {
			$allowedStatusForAsset = [];
		}

		return AssetStatus::whereIn('slug', $allowedStatusForAsset)->orWhere('slug', $statusSlug)->get();
	}

	/**
	 * Validates if the given asset status can be updated from the given from status to the given to status.
	 *
	 * @param string $from The slug of the current status of the asset.
	 * @param string $to The slug of the status to be updated to.
	 * @return bool Whether the asset status can be updated from the given from status to the given to status.
	 */
	public function validateStatusUpdate($from, $to)
	{
		$availableStatuses = config('asset-status.' . $from);

		return in_array($to, $availableStatuses);
	}

	/**
	 * Check if an asset can be moved.
	 *
	 * @param Asset $asset The asset model instance.
	 * @return boolean Whether the asset can be moved.
	 */
	public function canBeMoved($asset)
	{
		return $asset->assetStatus?->has_location;
	}

	/**
	 * Checks if an asset can be returned based on its status.
	 *
	 * @param Asset $asset The asset to be checked.
	 * @return bool True if the asset can be returned, otherwise false.
	 */
	public function canBeReturned($asset)
	{
		return $asset->assetStatus?->has_user;
	}

	/**
	 * The function checks if the from location of an asset is frozen or not.
	 * 
	 * @param asset 
	 * 
	 * @return true or false
	 */
	public function validateFromLocationFreeze($asset)
	{
		$asset = $asset->first();

		return $asset->location ? (optional($asset->location)->is_freeze == 0 ? true : false) : true;
	}
}
