<?php

namespace App\Services\Asset;

use App\Services\AssetHistory;
use App\Models\Asset;
use App\Events\BulkUpdates;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\Events\UpdateAssetStatus;
use Facades\App\Services\JiraDescription;


class AssetStatusHistoryService extends AssetHistory
{
    /**
     * Creates an asset history record for the given asset.
     *
     * @param mixed $status The new status of the asset.
     * @param array $data The additional data related to the asset history.
     * @param App\Models\Asset $asset The asset for which the history is being created.
     * @param array $inputData Input data containing ticket number and comments.
     * @param string|null $section The section of the asset history.
     * @return bool
     */
    public function createAssetHistory($status, $data, $asset, $inputData, $section = null)
    {
        $description = $this->getStatusDescription($status, $data, $asset, $inputData['comments'] ?? '');
        $userId = Auth::id();
        $assetHistory = [
            'user_id' => $userId,
            'asset_id' => $asset->id,
            'ticket_no' => $inputData['ticket_no'] ?? '',
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'status_updated',
            'comments' => $inputData['comments'] ?? '',
            'old_value' => optional($asset->assetStatus)->name,
            'new_value' => $status,
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        $assetHistory = $this->getHistoryData($assetHistory, $data, $asset);

        // For bulk upload we just save the asset history no jira comments.
        if ($section === 'bulk_upload') {
            event(new BulkUpdates($assetHistory));

            return true;
        }
        $jiraDescription = JiraDescription::getStatusDescription($status, $data, $asset, $inputData['comments'] ?? '');

        event(new UpdateAssetStatus($jiraDescription, $inputData['ticket_no'], $assetHistory, $userId));

        return true;
    }

    /**
     * Populate the asset history array with the additional data from the given asset data
     * and asset model.
     *
     * @param array $assetHistory The asset history data array.
     * @param array $assetData The asset data array.
     * @param Asset $asset The asset model instance.
     * @return array The updated asset history data array.
     */
    public function getHistoryData($assetHistory, $assetData, $asset)
    {
        $keyArray = ['loaner_return_date', 'location_id', 'user_id', 'loaner_retention_date', 'asset_status_id'];
        foreach ($keyArray as $key) {
            if (array_key_exists($key, $assetData)) {
                $assetHistory['new_' . $key] = $assetData[$key];
                $assetHistory['old_' . $key] = $asset->$key;
            }
        }
        if (array_key_exists('wipe_confirmation', $assetData)) {
            $assetHistory['wipe_confirmation'] = $assetData['wipe_confirmation'];
        }

        return $assetHistory;
    }

    /**
     * Create asset history for eol certificate upload
     *
     * @param App\Models\Asset $asset
     * @param int $newStatusId
     */
    public function createEolCertificateUploadHistory($asset, $newStatusId)
    {
        $assetHistory = [
            'asset_id' => $asset->id,
            'user_id' => Auth::id(),
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $newStatusId,
            'action' => 'certificate_uploaded',
            'description' => __('history.CertificateUploaded', [
                'asset_name' => $asset->serial_no,
                'asset_id' => $asset->id
            ]),
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
    }

    /**
     * Create asset history for asset status change from pending return to lost or stolen.
     *
     * @param App\Models\Asset $asset The asset model instance.
     * @param App\Models\AssetStatus $status The new asset status.
     * 
     * @return void
     */
    public function createPendingReturnAutoUpdateHistory($asset, $status)
    {
        $comments = "user terminated.";
        $description = __('history.Description', [
			'assetname'       => $asset->serial_no,
			'assetid'         => $asset->id,
			'oldasset_status' => optional($asset->assetStatus)->name,
			'newstatus'       => $status->name,
			'oldusername'     => ' from ' . optional($asset->user)->username,
			'roomname'        => ' to ' . optional($asset->user)->username,
			'commenttext'     => ' with comments ' . $comments

		]);

        $assetHistory = [
            'asset_id' => $asset->id,
            'action' => 'status_updated',
            'old_value' => optional($asset->assetStatus)->name,
            'new_value' => $status->name,
            'old_user_id' => $asset->user_id,
            'new_user_id' => $asset->user_id,
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $status->id,
            'description' => $description,
            'comments' => $comments,
            'created_at' => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at' => Carbon::now()->format('Y-m-d H:i:s'),
            'created_by' => 'Teqtivity Automation'
        ];

        event(new BulkUpdates($assetHistory));
    }
}
