<?php

namespace App\Services\Asset;

use Illuminate\Support\Facades\DB;

class AssetDataSortService
{

    /**
     * Applies the sorting rules to the given asset query.
     *
     * Looks up the given column in the join mappings and applies the specified
     * joins and orders. If no join mapping is found, looks up the given column
     * in the special orders and applies the specified closure. If no special order
     * is found, sorts by the given column if it is one of 'modified_by' or
     * 'modified_date', and by the 'id' column otherwise.
     *
     * @param  \Illuminate\Database\Eloquent\Builder $assets
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function applySorting($assets)
    {
        $columns = array_column(request()->input('columns'), 'data');
        $orderColumn = request()->input('order.0.column');
        $order = $columns[$orderColumn];
        $dir = request()->input('order.0.dir');

        $joinMappings = $this->getJoinMappings($dir);

        $specialOrders = $this->getSpecialOrders($dir);

        // Apply mapped joins and orders
        if (isset($joinMappings[$order])) {
            $mapping = $joinMappings[$order];
            $assets = $this->applyJoinMappings($assets, $mapping);

            $assets = $this->applyOrderMappings($assets, $mapping);
        } elseif (isset($specialOrders[$order])) {
            $assets = $specialOrders[$order]($assets, $dir);
        } else if (in_array($order, ['modified_by', 'modified_date'])) {
            $assets = $this->sortByAssetHistoryFields($assets, $order, $dir);
        } else {
            $assets = $assets->orderBy('id', $dir);
        }

        return $assets;
    }

    /**
     * Retrieve the join mappings for asset sorting, replacing the direction placeholder.
     *
     * This function fetches the join mappings configuration and replaces any
     * placeholder '{dir}' in the order fields with the actual direction specified
     * by the $dir parameter. The mappings define how assets should be joined and
     * ordered based on various criteria.
     *
     * @param string $dir The direction ('asc' or 'desc') to apply to the order fields.
     * 
     * @return array The modified join mappings with the direction applied.
     */

    private function getJoinMappings($dir)
    {
        $mapping = config('asset-sort-order');

        foreach ($mapping as &$item) {
            // Replace '{dir}' with the actual $dir value
            if (isset($item['order'])) {
                if (is_array($item['order'][0])) {
                    foreach ($item['order'] as &$subOrder) {
                        $subOrder[1] = str_replace('{dir}', $dir, $subOrder[1]);
                    }
                } else {
                    $item['order'][1] = str_replace('{dir}', $dir, $item['order'][1]);
                }
            }
        }

        return $mapping;
    }

    /**
     * Retrieves the special orders configuration for asset sorting, which
     * specifies closures that sort the assets by specific criteria.
     *
     * @param string $dir The direction ('asc' or 'desc') to apply to the order fields.
     * 
     * @return array The special orders configuration.
     */
    private function getSpecialOrders($dir)
    {
        return [
            'created_at' => fn($query, $dir) => $query->orderBy('assets.created_at', $dir === 'asc' ? 'desc' : 'asc'),
            'loaner_return_date' => fn($query, $dir) => $query->orderBy('assets.loaner_return_date', $dir === 'asc' ? 'desc' : 'asc'),
            'serial_no' => fn($query, $dir) => $query->orderBy('serial_no', $dir === 'asc' ? 'desc' : 'asc'),
            'asset_tag' => fn($query, $dir) => $query->orderBy('asset_tag', $dir === 'asc' ? 'desc' : 'asc'),
            'age' => fn($query, $dir) => $query->orderBy('created_at', $dir === 'asc' ? 'desc' : 'asc'),
        ];
    }

    /**
     * Sorts assets by asset history fields.
     *
     * This function joins the assets with their latest asset history entries and sorts them based on the specified order criteria.
     * If the order is 'modified_by', it sorts by user's first name, last name, and email. Otherwise, it defaults to sorting by
     * the asset history's creation date.
     *
     * @param \Illuminate\Database\Eloquent\Builder $assets The query builder instance for assets.
     * @param string $order The field to order by, such as 'modified_by'.
     * @param string $dir The direction ('asc' or 'desc') to apply to the order.
     * 
     * @return \Illuminate\Database\Eloquent\Builder The updated query builder with the sorting applied.
     */
    private function sortByAssetHistoryFields($assets, $order, $dir)
    {
        $assets = $assets->leftJoin("asset_histories", function ($join) {
            $join->on('asset_histories.asset_id', '=', 'assets.id')
                ->on('asset_histories.id', '=', DB::raw("(SELECT MAX(id) FROM asset_histories WHERE asset_histories.asset_id = assets.id)"));
        });

        if ($order == 'modified_by') {
            // Sort by user's first name, last name, and email
            return $assets->leftJoin('users', 'asset_histories.user_id', '=', 'users.id')
                ->orderBy('users.first_name', $dir)
                ->orderBy('users.last_name', $dir)
                ->orderBy('users.email', $dir);
        }

        // Default sort by asset_histories.created_at
        return $assets->orderBy('asset_histories.created_at', $dir);
    }

    /**
     * Applies join mappings to a query builder instance.
     *
     * @param \Illuminate\Database\Eloquent\Builder $assets The query builder instance for assets.
     * @param array $mapping The join mapping configuration.
     *
     * @return \Illuminate\Database\Eloquent\Builder The updated query builder with the join mappings applied.
     */
    private function applyJoinMappings($assets, $mapping)
    {
        if (isset($mapping['join'])) {
            $assets = $assets->leftJoin(...$mapping['join']);
        }

        if (isset($mapping['joins'])) {
            foreach ($mapping['joins'] as $join) {
                $assets = $assets->leftJoin(...$join);
            }
        }

        return $assets;
    }

    /**
     * Applies order mappings to the assets query.
     *
     * @param \Illuminate\Database\Eloquent\Builder $assets The assets query to be ordered.
     * @param array $mapping The mapping array containing order rules.
     * 
     * @return \Illuminate\Database\Eloquent\Builder The updated query builder with the order applied.
     */

    private function applyOrderMappings($assets, $mapping)
    {
        if (isset($mapping['order'])) {
            if (count($mapping['order']) === 2 && is_string($mapping['order'][0]) && is_string($mapping['order'][1])) {
                $assets = $assets->orderBy($mapping['order'][0], $mapping['order'][1]);
            } else {
                foreach ((array)$mapping['order'] as $orderRule) {
                    if (is_array($orderRule) && count($orderRule) === 2) {
                        $assets = $assets->orderBy($orderRule[0], $orderRule[1]);
                    }
                }
            }
        }

        return $assets;
    }
}
