<?php

namespace App\Services\Asset;

use App\Http\Responses\ReportOutputData;
use App\Repositories\AssetRepository;

class AssetDataService
{
    public function __construct(
        protected AssetRepository $assetRepository,
        protected AssetDataSortService $assetSortService,
        protected AssetDataFilterService $assetFilterService,
        protected ReportOutputData $reportOutputData
    ) {}

    /**
     * Retrieve a list of assets based on the input data from the form.
     *
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function getAssets()
    {
        $form = request('form', []);

        $assets = $this->assetRepository->getAssetsQueryWithRelations();

        $assets = $this->assetFilterService->filterAssets($assets, $form);

        $assets = $this->assetSortService->applySorting($assets);

        return $this->reportOutputData->applyPagination($assets);
    }

    /**
     * Processes a collection of assets and retrieves their data in a nested format.
     *
     * @param \Illuminate\Database\Eloquent\Collection $assets Collection of assets to process.
     * @param int $start Starting index for processing assets.
     * @param array $data The array to which the nested data of assets will be appended.
     *
     * @return array The array containing the nested data of processed assets.
     */
    public function getAssetData($assets, $start, $data = [])
    {
        $parentIndex = $start;

        foreach ($assets as $asset) {
            if (!$asset->parentAsset) {
                $parentIndex++;

                $assetLink = $this->getAssetLink($asset);
                $nestedData = $this->getNestedData($asset, $parentIndex, '', '', $assetLink);
                $data[] = $nestedData;

                $this->processChildren($asset, $parentIndex, $data);
            }
        }

        return $data;
    }

    /**
     * Recursively processes children assets and appends their nested data to the given array.
     *
     * @param  \App\Models\Asset  $asset  The parent asset.
     * @param  int  $parentIndex  The parent index.
     * @param  array  &$data  The array to which the nested data of children assets will be appended.
     *
     * @return void
     */
    private function processChildren($asset, $parentIndex, &$data)
    {
        if (!is_countable($asset->childrenAsset) || count($asset->childrenAsset) === 0) {
            return;
        }

        foreach ($asset->childrenAsset as $index => $child) {
            $itemClass = $index === count($asset->childrenAsset) - 1 ? 'last' : '';
            $assetLink = $this->getAssetLink($child);
            $nestedData = $this->getNestedData($child, $parentIndex . '.' . ($index + 1), $itemClass, $asset->id, $assetLink);
            $data[] = $nestedData;
        }
    }

    /**
     * Generates a URL link for the given asset based on its asset type.
     *
     * If the asset type is related to mobile assets, a mobile asset route is generated;
     * otherwise, a general asset route is generated.
     *
     * @param \App\Models\Asset $asset The asset for which the link is generated.
     * @return string The generated URL link for the asset.
     */
    private function getAssetLink($asset)
    {
        if ($asset->assetType) {
            return $asset->assetType->hasRelationToSlug('mobile_assets')
                ? route('mobile-assets.show', $asset->id)
                : route('assets.show', $asset->id);
        } else {
            return route('assets.show', $asset->id);
        }
    }

    /**
     * Generates nested data for a given asset and index.
     *
     * @param  \App\Models\Asset  $asset  The asset object.
     * @param  int|string  $index  The index of the nested data.
     * @param  string  $last  A string indicating whether the nested data is the last in its branch.
     * @param  int  $parent  The ID of the parent asset.
     * @param  string  $assetLink  The URL link to the given asset.
     *
     * @return array  The nested data.
     */
    public function getNestedData($asset, $index, $last, $parent, $assetLink)
    {
        $nestedData = [
            'dataLast' => $last,
            'dataId' => $asset->id,
            'dataParent' => $parent,
            'id' => $index,
            'type' => $asset->assetType?->name ?? '',
            'asset_tag' => "<a href=$assetLink>$asset->asset_tag</a>",
            'serial_no' => "<a href=$assetLink>$asset->serial_no</a>",
            'hardware_standard' => $asset->makeAndModel?->makeModelName ?? '',
            'technical_spec' => $asset->technicalSpec?->details ?? '',
            'status' => $asset->assetStatus?->name ?? '',
            'user_location' => $this->getUserLocation($asset),
            'carrier' => $asset->carrier?->name ?? '',
            'loaner_return_date' => $asset->loaner_return_date,
            'lease_start_date' => $asset->lease_start_date,
            'lease_end_date' => $asset->lease_end_date,
            'age' => $asset->assetAge,
            'modified_date' => $asset->latestAssetHistory?->updated_at ?? '',
            'modified_by' => $this->getModifiedBy($asset),
        ];

        return $nestedData;
    }

    /**
     * Generates a string representing the user's location for a given asset.
     *
     * This method constructs a user location string by creating a link to the user
     * and, if available, appends the location link of the asset. If both user and
     * location are present, they are separated by a " / ".
     *
     * @param \App\Models\Asset $asset The asset object containing user and location data.
     * @return string A string containing HTML links to the user and/or location.
     */
    private function getUserLocation($asset)
    {
        $userLocation = generateUserLink($asset->user?->id, $asset->user?->user_name);

        if ($asset->location) {
            $userLocation .= (($asset->user) ? ' / ' : '') . generateLocationLink($asset->location?->id, $asset->location?->room_name);
        }

        return $userLocation;
    }

    /**
     * Generates a link to the user who last modified the asset.
     *
     * @param \App\Models\Asset $asset The asset object containing modification history.
     * @return string A string containing an HTML link to the last modifying user.
     */
    private function getModifiedBy($asset)
    {
        return generateUserLink($asset->latestAssetHistory?->user?->id, $asset->latestAssetHistory?->user?->user_name);
    }
}
