<?php

namespace App\Services\Asset;

use App\Models\Asset;
use App\Repositories\CommonFilterRepository;

class AssetDataFilterService
{
    public function __construct(
        protected CommonFilterRepository $commonFilterRepository,
    ) {}

    /**
     * Filters the assets based on the provided input data.
     *
     * @param mixed $assets The initial asset query.
     * @param array $inputData The input data containing filter criteria.
     * 
     * @return mixed The filtered asset query.
     */

    public function filterAssets($assets, $form)
    {
        $inputData = $this->getInputData($form);

        // Apply category-specific filters
        $assets = $this->applyCategoryFilters($assets, $inputData['category'] ?? '');

        $assets = $this->applyDirectFilter($assets, $inputData);


        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'makeAndModel', 'manufacturer_id', $inputData['manufacturer'] ?? '');

        $assets = $this->commonFilterRepository->filterWithWhereHasRelationFields($assets, 'location', 'building', $inputData['building'] ?? '');

        // Asset Age Filter
        if (!empty($inputData['asset_age_value']) && !empty($inputData['asset_age_unit'])) {
            $assets = $this->commonFilterRepository->applyAgeFilter($assets, $inputData['asset_age_value'], $inputData['asset_age_unit']);
        }

        $assets = $this->applyDateRangeFilter($assets, $inputData);

        return $assets;
    }

    /**
     * Retrieves input data from the form.
     * @param array $form The form data.
     *
     * @return array The input data with default values for missing keys.
     */
    private function getInputData($form)
    {
        return [
            'category'                => $form['category'] ?? '',
            'asset_type'              => $form['asset_type'] ?? '',
            'status'                  => $form['status'] ?? '',
            'carrier'                 => $form['carrier'] ?? '',
            'make_and_model'          => $form['make_and_model'] ?? '',
            'technical_spec'          => $form['technical_spec'] ?? '',
            'location'                => $form['location'] ?? '',
            'user'                    => $form['user'] ?? '',
            'asset_age_value'         => $form['asset_age_value'] ?? null,
            'asset_age_unit'          => $form['asset_age_unit'] ?? null,
            'lease_start_date_from'   => $form['lease_start_date_from'] ?? '',
            'lease_start_date_to'     => $form['lease_start_date_to'] ?? '',
            'lease_end_date_from'     => $form['lease_end_date_from'] ?? '',
            'lease_end_date_to'       => $form['lease_end_date_to'] ?? '',
            'date_deployed_from'      => $form['date_deployed_from'] ?? '',
            'date_deployed_to'        => $form['date_deployed_to'] ?? '',
            'loaner_return_date_from' => $form['loaner_return_date_from'] ?? '',
            'loaner_return_date_to'   => $form['loaner_return_date_to'] ?? '',
            'lost_date_from'          => $form['lost_date_from'] ?? '',
            'lost_date_to'            => $form['lost_date_to'] ?? '',
            'building'                => $form['building'] ?? '',
            'asset_tag'               => $form['asset_tag'] ?? '',
            'serial_no'               => $form['serial_no'] ?? '',
            'manufacturer'            => $form['manufacturer'] ?? '',
        ];
    }

    /**
     * Apply category-specific filters to the asset query.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query  The asset query to be filtered.
     * @param  string  $category  The category to filter by.
     *
     * @return \Illuminate\Database\Eloquent\Builder  The filtered asset query.
     */
    private function applyCategoryFilters($query, $category)
    {
        switch ($category) {
            case 'mobile':
                return $query->mobileAsset();
            case 'server':
                return $query->serverAsset()->doesnthave('parentAsset');
            case 'network':
                return $query->networkAsset();
            case 'avassets':
                return $query->avAsset();
            case 'research':
                return $query->researchAsset();
            case 'it':
                return $query->regularAsset();
            default:
                return $query;
        }
    }

    /**
     * Applies filters to the query based on direct fields from the input data.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query  The asset query to be filtered.
     * @param  array  $inputData  The input data used to filter the assets.
     *
     * @return \Illuminate\Database\Eloquent\Builder  The filtered asset query.
     */
    private function applyDirectFilter($query, $inputData)
    {
        // Define relation filters
        $relationFilters = [
            'asset_type_id' => $inputData['asset_type'] ?? '',
            'asset_status_id' => $inputData['status'] ?? '',
            'carrier_id' => $inputData['carrier'] ?? '',
            'make_and_model_id' => $inputData['make_and_model'] ?? '',
            'technical_spec_id' => $inputData['technical_spec'] ?? '',
            'assets.location_id' => $inputData['location'] ?? '',
            // 'assets.location.building' => $inputData['building'] ?? '',
            'assets.user_id' => $inputData['user'] ?? '',
            'serial_no' => $inputData['serial_no'] ?? '',
            'asset_tag' => $inputData['asset_tag'] ?? '',
        ];

        foreach ($relationFilters as $column => $value) {
            if ($column === 'asset_tag' && !empty($inputData['asset_tag'])) {
                $value = $this->getParentTags($inputData['asset_tag']);
            }

            $this->commonFilterRepository->filterWithDirectFields($query, $column, $value);
        }

        return $query;
    }

    /**
     * Applies date range filters to the asset query based on the input data.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query The asset query to be filtered.
     * @param array $inputData The input data containing date range filter criteria.
     *
     * @return \Illuminate\Database\Eloquent\Builder The filtered asset query.
     */

    private function applyDateRangeFilter($query, $inputData)
    {
        // Date Filters
        $dateFilters = [
            'lease_start_date' => ['from' => $inputData['lease_start_date_from'] ?? null, 'to' => $inputData['lease_start_date_to'] ?? null],
            'lease_end_date' => ['from' => $inputData['lease_end_date_from'] ?? null, 'to' => $inputData['lease_end_date_to'] ?? null],
            'date_deployed' => ['from' => $inputData['date_deployed_from'] ?? null, 'to' => $inputData['date_deployed_to'] ?? null],
            'loaner_return_date' => ['from' => $inputData['loaner_return_date_from'] ?? null, 'to' => $inputData['loaner_return_date_to'] ?? null],
            'lost_date' => ['from' => $inputData['lost_date_from'] ?? null, 'to' => $inputData['lost_date_to'] ?? null],
        ];

        foreach ($dateFilters as $column => $range) {
            $this->commonFilterRepository->filterWithDirectDateRange($query, $column, $range['from'], $range['to']);
        }

        return $query;
    }

    /**
     * Given an asset tag (or array of tags), this function will return an array
     * of the given tags, plus any parent tags associated with the given tags.
     * If a tag is not found in the db, it will be ignored.
     *
     * @param string|array $assetTag The asset tag(s) to get the parent tags for.
     * @return array The given tags, plus their parent tags.
     */
    public function getParentTags($assetTag)
    {
        if (!is_array($assetTag)) {
            $assetTag = [$assetTag];
        }

        $parentTags = Asset::whereIn('id', Asset::whereIn('asset_tag', $assetTag)->pluck('parent_asset_id'))->pluck('asset_tag')->toArray();

        $assetTag = array_filter(array_merge($assetTag, $parentTags));

        return $assetTag;
    }
}
