<?php

namespace App\Services\Asset;

use App\Http\Traits\BulkUploadTrait;
use Facades\App\Repositories\BulkUpload;
use Illuminate\Support\Facades\Validator;
use App\Models\MakeAndModel;
use App\Models\AssetTab;
use App\Rules\CheckLocationFreezeRule;
use App\Services\Asset\CreateAssetService;
use Illuminate\Validation\Rule;

class AssetBulkCreateService
{
    use BulkUploadTrait;

    public function __construct(protected CreateAssetService $createAssetService) {}

    /**
     * Retrieves the view data for a given path, file name and type.
     *
     * @param string $path   The path of the file.
     * @param int    $count  The count of the file.
     * @param string $type   The type of upload.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the view data.
     */
    public function getViewData($path, $count, $type = 'it_assets')
    {
        try {
            $data = $this->importAssetData($path, $count, $type);
        } catch (\Exception) {
            $errors = 'Invalid file.';
            return ['errors' => view('assets.partials.upload-csv-errors', compact('errors'))->render()];
        }

        if ($view = $this->handleImportErrors($data)) {
            return response()->json($view);
        }

        $csvData = $data['csvData'];

        if ($view = $this->handleCsvColumnErrors($path, $csvData, $this->getHeaderMap($type))) {
            return response()->json($view);
        }

        if ($csvData) {

            $validatedData = $this->validateFields($csvData, $type);

            $errors = $validatedData['errors'];
            $csvData = $validatedData['csvData'];

            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            $duplicationValidatedData = $this->checkRepeatUploadData($csvData);
            $csvData = $duplicationValidatedData['csvData'];
            $duplicateErrors = $duplicationValidatedData['errors'];
            $view['duplicate_errors'] = view('assets.partials.upload-duplicate-errors', compact('duplicateErrors'))->render();

            if (!empty($duplicateErrors)) {
                $csvData = [];
            }

            $view['data'] = view('assets.create.partials.upload-assets', compact('csvData'))->render();
            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    /**
     * @param string $path
     * @param int $count
     * @param string $type
     * @return array
     */
    public function importAssetData($path, $count, $type = 'it_assets')
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];

        if (empty($items)) {
            return $csvData;
        }

        foreach ($items as  $item) {
            if (!count($item)) {
                continue;
            } //skip empty columns

            $csvData['csvData'][] = $this->generateCSVData($item, ++$count, $type);
        }

        return $csvData;
    }

    /**
     * Process a single row of csv data into an array of asset data
     * @param array $item A single row of csv data
     * @param int $count The current count of the csv data
     * @param string $type The type of asset being imported
     * @return array The processed asset data
     */
    public function generateCSVData($item, $count, $type)
    {
        if (empty($item)) {
            return false;
        }

        $assetData    = [];

        foreach ($this->getHeaderMap($type) as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField, $fileHeader);

            if ($result !== false) {
                $item[$fileHeader] = $result;
            }

            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "make_and_model_id" && !empty($item[$fileHeader])) {
                $makeModelData = $this->processMakeAndModel($item[$fileHeader]);
                $assetData = array_merge($assetData, $makeModelData);
            }

            if ($dbField == "asset_tag") {
                $assetData['asset_tag'] = $this->createAssetService->getAssetTag($item[$fileHeader]);
            }
        }

        $assetData['count_add'] = $count;

        // Used in bulk upload trait, duplicate check validation.
        $assetData['asset_count'] = $count;
        session(['count_add' => $assetData['count_add']]);

        return compact('assetData');
    }

    /**
     * Retrieve the header map for bulk uploads, based on the given type.
     *
     * @param string $type The type of asset being imported.
     * @return array The header map configuration.
     */
    public function getHeaderMap($type = null)
    {
        if ($type == 'mobile_assets') {
            return config('bulk-upload.mobileAssetData');
        }

        return config('bulk-upload.itAssetData');
    }

    /**
     * Retrieves relational values for specified fields from the item.
     *
     * @param array  $item       The item data from the CSV file.
     * @param string $dbField    The database field to be processed.
     * @param string $fileHeader The corresponding file header from the CSV.
     *
     * @return mixed The relational value for the field, or false if no value is found.
     */
    private function getRelationalValues($item, $dbField, $fileHeader)
    {
        $mapping = [
            'location_id' => 'getLocationId',
            'technical_spec_id' => 'getTechnicalSpecId',
            'vendor_id' => 'getVendorId',
            'carrier_id' => 'getCarrierId',
            'parent_asset_id' => 'getParentAssetId',
        ];

        if (isset($mapping[$dbField])) {
            $method = $mapping[$dbField];
            $value = BulkUpload::$method($item, $dbField, $fileHeader);

            // If the value is empty but $fileHeader exists in $item, return 'invalid--id'
            return empty($value) && !empty($item[$fileHeader]) ? 'invalid--id' : $value;
        }

        return false;
    }


    /**
     * Retrieves the make and model details for a given make and model slug.
     *
     * @param string $makeAndModelSlug The slug of the make and model.
     * @return array The make and model details, including asset type ID, manufacturer ID,
     *     and make and model ID. If the make and model is not found, returns an empty array
     *     with null values.
     */
    protected function processMakeAndModel($makeAndModelSlug)
    {
        $makeModel = MakeAndModel::where('slug', trim(str_slug($makeAndModelSlug, '_')))->first();

        if ($makeModel) {
            return [
                'asset_type_id' => $makeModel->asset_type_id,
                'manufacturer_id' => $makeModel->manufacturer_id,
                'make_and_model_id' => $makeModel->id,
            ];
        }

        return [
            'asset_type_id' => 'invalid--id',
            'manufacturer_id' => 'invalid--id',
            'make_and_model_id' => 'invalid--id',
        ];
    }

    /**
     * Validate each row of the CSV
     * 
     * @param array $data The data from the current row of the CSV
     * @param int $count The current row number
     * @param string $type The type of upload
     * 
     * @return object
     */
    public function csvValidator($data, $count, $type)
    {
        $allowedAssetTypes = AssetTab::query()->assetTypeIdsBySlug($type);

        $validator = Validator::make(
            $data,
            [
                'location_id'       => ['required', 'exists:locations,id', new CheckLocationFreezeRule($count)],
                'asset_type_id'     => [$data['make_and_model_id'] && is_int($data['make_and_model_id']) ? 'required' : 'nullable', 'exists:asset_types,id', ($data['asset_type_id'] && is_int($data['asset_type_id'])) ? Rule::in($allowedAssetTypes) : ''],
                'asset_tag'         => ['required', 'unique:assets,asset_tag'],
                'po_id'             => ['required'],
                'vendor_id'         => ['nullable', 'exists:vendors,id'],
                'serial_no'         => ['required', 'unique:assets,serial_no'],
                // 'ticket_no'         => ['required'],
                'make_and_model_id' => ['required', 'exists:make_and_models,id'],
                'technical_spec_id' => ['required', 'exists:technical_specs,id'],
                'carrier_id'        => ['required_if:asset_type_slug,mobile_phone,mobile_phone_number', 'exists:carriers,id'],
                'imei'              => ['required_if:asset_type_slug,mobile_phone'],
                'lease_start_date'  => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'lease_end_date'    => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'warranty_end_date' => ['nullable', 'date_format:' . config('date.formats.read_date_format')],

            ],
            [
                'location_id.required'                 => 'Line no ' . $count . ' : The location is required.',
                'location_id.exists'                   => 'Line no ' . $count . ' : The location does not exist.',
                'asset_type_id.required'               => 'Line no ' . $count . ' : The asset type is required.',
                'asset_type_id.in'                     => 'Line no ' . $count . ' : The asset type is not allowed.',
                'asset_type_id.exists'                 => 'Line no ' . $count . ' : The asset type does not exist.',
                'asset_tag.required'                   => 'Line no ' . $count . ' : The asset tag # is required.',
                'asset_tag.unique'                     => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'po_id.required'                       => 'Line no ' . $count . ' : The PO # is required.',
                'vendor_id.exists'                     => 'Line no ' . $count . ' : The vendor does not exist.',
                'serial_no.required'                   => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.unique'                     => 'Line no ' . $count . ' : The serial # has already been taken.',
                'ticket_no.required'                   => 'Line no ' . $count . ' : The ticket # is required.',
                'make_and_model_id.required'           => 'Line no ' . $count . ' : The hardware standard is required.',
                'make_and_model_id.exists'             => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.required'           => 'Line no ' . $count . ' : The Technical Spec is required.',
                'technical_spec_id.exists'             => 'Line no ' . $count . ' : The Technical Spec does not exist.',
                'make_and_model_id.required_unless'    => 'Line no ' . $count . ' : The hardware standard does not exist.',
                'technical_spec_id.required_unless'    => 'Line no ' . $count . ' : The technical specs does not exist.',
                'carrier_id.required_if'               => 'Line no ' . $count . ' : The carrier is required.',
                'carrier_id.exists'                    => 'Line no ' . $count . ' : The carrier does not exist.',
                'imei.required_if'                     => 'Line no ' . $count . ' : IMEI is required for Mobile Phone.',
                'lease_start_date.required_if'         => 'Line no ' . $count . ' : The lease start date is required for Mobile Phone.',
                'lease_start_date.date_format'         => 'Line no ' . $count . ' : The lease start date format is invalid. Please use ' . config('date.formats.read_date_format') . ' format.',
                'lease_end_date.required_if'           => 'Line no ' . $count . ' : The lease end date is required Mobile Phone.',
                'lease_end_date.date_format'           => 'Line no ' . $count . ' : The lease end date format is invalid. Please use ' . config('date.formats.read_date_format') . ' format.',
                'warranty_end_date.date_format'        => 'Line no ' . $count . ' : The warranty end date format is invalid. Please use ' . config('date.formats.read_date_format') . ' format.',


            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Retrieve the view data for bulk upload based on the path, upload data, and count
     * 
     * @param string $path The path of the file
     * @param array $uploadData The data from the upload form
     * @param int $count The count of the file
     * 
     * @return json The JSON response containing the view data
     */
    public function getBulkUploadViewData($path, $uploadData, $count)
    {
        try {
            $data = $this->importBulkUploadData($path, $uploadData, $count);
        } catch (\Exception) {
            $errors = 'Invalid file.';
            return ['errors' => view('assets.partials.upload-csv-errors', compact('errors'))->render()];
        }

        if ($view = $this->handleImportErrors($data)) {
            return response()->json($view);
        }

        $csvData = $data['csvData'];

        if ($view = $this->handleCsvColumnErrors($path, $csvData, $this->getBulkUploadHeaderMap($uploadData['type']))) {
            return response()->json($view);
        }

        if ($csvData) {

            $validatedData = $this->validateUploadFields($csvData, $uploadData['type']);

            $errors = $validatedData['errors'];
            $csvData = $validatedData['csvData'];

            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            $duplicationValidatedData = $this->checkRepeatUploadData($csvData);
            $csvData = $duplicationValidatedData['csvData'];
            $duplicateErrors = $duplicationValidatedData['errors'];
            $view['duplicate_errors'] = view('assets.partials.upload-duplicate-errors', compact('duplicateErrors'))->render();

            if (!empty($duplicateErrors)) {
                $csvData = [];
            }

            $view['data'] = view('assets.create.partials.upload-assets', compact('csvData'))->render();
            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    /**
     * @param string $path
     * @param array $contentData
     * @param int $count
     * @return array
     */
    public function importBulkUploadData($path, $contentData, $count)
    {
        $data = $this->getItems($path);
        $items = $data['csvData'];
        $csvData['error'] = $data['error'];
        $csvData['csvData'] = [];
        if (!empty($items)) {
            foreach ($items as $item) {
                if (!count($item)) {
                    continue;
                } //skip empty columns
                $count++;
                $csvData['csvData'][] = $this->generateUploadCSVData($item, $count, $contentData);
            }
        }

        return $csvData;
    }

    /**
     * Generates asset data from a single row of CSV data.
     *
     * This function processes a single CSV row to map its fields to the 
     * database fields as defined in the header map. It also generates a 
     * unique asset tag using the `createAssetService`.
     *
     * @param array $item A single row of CSV data.
     * @param int $count The current count of processed rows.
     * @param array $data Additional data that includes the asset type and other attributes.
     * @return array|bool An array containing the processed asset data or false if the item is empty.
     */
    public function generateUploadCSVData($item, $count, $data)
    {
        if (empty($item)) {
            return false;
        }

        $headerMap = $this->getBulkUploadHeaderMap($data['type']);

        $assetData    = $data;

        foreach ($headerMap as $dbField => $fileHeader) {

            $assetData[$dbField] = !empty($item[$fileHeader]) ? $item[$fileHeader] : null;

            if ($dbField == "asset_tag") {
                $assetData['asset_tag'] = $this->createAssetService->getAssetTag($item[$fileHeader]);
            }
        }

        $assetData['count_add'] = $count++;
        // Used in bulk upload trait, duplicate check validation.
        $assetData['asset_count'] = $assetData['count_add'];
        session(['count_add' => $assetData['count_add']]);

        return compact('assetData');
    }

    /**
     * Retrieve the header map for bulk uploads, based on the given type.
     *
     * @param string $type The type of asset being imported.
     * @return array The header map configuration.
     */
    private function getBulkUploadHeaderMap($type)
    {
        if ($type == 'mobile_assets') {

            return config('bulk-upload.uploadAssetMobileData');
        }

        return config('bulk-upload.uploadAssetItData');
    }

    /**
     * Validate each row of the CSV data for upload.
     *
     * @param array $csvData The CSV data to validate.
     * @param string $type The type of asset being imported.
     * @return array The validation errors, or an empty array if all rows are valid.
     */
    public function validateUploadFields($csvData, $type)
    {
        $count = 2;
        $errors = [];
        $validData = [];
        foreach ($csvData as $data) {
            $error = $this->csvUploadValidator($data['assetData'], $type, $count++);
            if ($error !== null) {
                $errors[] = $error;
                continue;
            }
            $validData[] = $data;
        }

        return ['errors' => $errors, 'csvData' => $validData]; //$errors;
    }

    /**
     * Validate a single row of the CSV data for upload.
     *
     * @param array $data The row of data from the CSV.
     * @param string $type The type of asset being imported.
     * @param int $count The line number of the error.
     * @return array|null The validation errors, or null if the row is valid.
     */
    private function csvUploadValidator($data, $type, $count)
    {

        $validator = Validator::make(
            $data,
            [
                'asset_tag' => ['required', 'unique:assets,asset_tag'],
                'serial_no' => ['required', 'unique:assets,serial_no'],
                'imei'      => [in_array($type, ['mobile_assets']) ? 'required' : 'nullable'],
            ],
            [
                'asset_tag.required' => 'Line no ' . $count . ' : The asset tag # is required.',
                'asset_tag.unique' => 'Line no ' . $count . ' : The asset tag # has already been taken.',
                'serial_no.required' => 'Line no ' . $count . ' : The serial # is required.',
                'serial_no.unique' => 'Line no ' . $count . ' : The serial # has already been taken.',
                'imei.required'    => 'Line no ' . $count . ' : IMEI is required.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }
}
