<?php

namespace App\Services\Asn\Presidio;

use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\MakeAndModel;
use App\Models\AssetHistory;
use App\Models\TechnicalSpecs;
use App\Models\AsnHardwareMapping;
use App\Services\Asn\AbstractAsnMappingService;
use App\Repositories\Asn\PresidioAssetsRepository;

class PresidioHardwareMappingService extends AbstractAsnMappingService
{
    /**
     * PresidioHardwareMappingService Constructor
     *
     * @param PresidioAssetsRepository $presidioRepository
     */
    public function __construct(protected PresidioAssetsRepository $presidioRepository)
    {
        $this->presidioRepository = $presidioRepository;
    }

    /**
     * Retrieve Presidio hardware mapping data.
     *
     * This function fetches the hardware mapping data for Presidio from the repository.
     *
     * @return Eloquent The Presidio hardware mappings data.
     */

    public function getPresidioMapping()
    {
        return $this->presidioRepository->getHardwareMappingData();
    }

    /**
     * Retrieve Presidio hardware mapping data as a collection.
     *
     * Retrieves Presidio hardware mapping data from the database as a collection.
     *
     * @return \Illuminate\Database\Eloquent\Collection The Presidio hardware mappings data.
     */
    public function getPresidioMappingData()
    {
        return $this->getPresidioMapping()->get();
    }

    /**
     * Filter presidio hardware mapping data
     *
     * @return Eloquent
     */
    public function filter()
    {
        $presidioMappingData = $this->getPresidioMapping();

        $make_and_model    = request('make_and_model');
        $technical_spec    = request('technical_spec');
        $searchText        = request('searchText');

        if ($make_and_model) {

            $presidioMappingData = $presidioMappingData->whereIn('asn_hardware_mappings.make_and_model_id', $make_and_model);
        }

        if ($technical_spec) {

            $presidioMappingData = $presidioMappingData->whereIn('asn_hardware_mappings.technical_spec_id', $technical_spec);
        }

        if ($searchText) {

            $presidioMappingData = $presidioMappingData->where('asn_hardware_mappings.part_no', 'like', '%' . $searchText . '%');
        }

        return  $presidioMappingData->get();
    }

    /**
     * Store mapping to system and update hardware details of related assets
     *
     * @return void
     */
    public function store()
    {
        $makeModel  = MakeAndModel::findOrFail(request('make_and_model'));
        $techSpec   = TechnicalSpecs::findOrFail(request('technical_spec'));

        $hardwareMapping    = AsnHardwareMapping::Create([
            'make_and_model_id' => $makeModel ? $makeModel->id : null,
            'technical_spec_id' => $techSpec ? $techSpec->id : null,
            'part_no'           => request('part_no'),
            'provider'          => 'presidio',
        ]);

        $this->updatePresidioAssets($hardwareMapping->part_no, $hardwareMapping->make_and_model_id, $hardwareMapping->technical_spec_id, ($makeModel->asset_type_id ?? null));
    }

    /**
     * Bulk upload Presidio hardware mappings from a CSV file.
     *
     * The CSV file should be in the format specified in the bulk-upload.asnHardwareMappingData config.
     * The function will return a message and an optional errors array.
     * If the file is not a CSV file or does not match the template, the function will return an error message.
     * If the import is successful, the function will return a success message.
     * If there are errors during the import, the function will return an error message and an array of errors.
     *
     * @return array
     */
    public function bulkUpload($request)
    {
        $file        = $request->file('file');
        $extension   = strtolower($file->getClientOriginalExtension());
        $path        = $file->storeAs('public/presidio_mapping_bulk_upload', 'presidio-hardware-mapping-' . date('m-d-y-h-i-s') . '.' . $extension);
        $path        = storage_path('app/' . $path);
        $message     = 'Bulk mapping completed successfully';
        $headerMap   = config('bulk-upload.asnHardwareMappingData');
        $checkHeader = $this->checkHeader($path, $headerMap);

        if ($checkHeader) {
            $errors = 'Uploaded file does not match the template.';

            return compact('errors');
        }

        $errors = $this->importHardwareMappingData($path, 'presidio');

        if ($errors) {
            $message = 'Mapping completed with errors';
        }

        return compact('message', 'errors');
    }

    /**
     * Update assets with given hardware details
     *
     * @param string $part_no  manufacturer part#
     * @param int $makeModelId  make_and_model id
     * @param int $techSpecId  technical specs id
     * @param int $assetTypeId  asset type id
     *
     * @return void
     */
    public function updatePresidioAssets($partNo, $makeModelId = null, $techSpecId = null, $assetTypeId = null)
    {
        $assetStatusId  = AssetStatus::where('slug', 'presidio_in_transit')->pluck('id');
        $makeModel      = MakeAndModel::findOrFail($makeModelId);
        $manufacturer   = optional($makeModel)->manufacturer ?? null;

        $assets = Asset::where([['asset_status_id', $assetStatusId], ['description', $partNo]])->get();
        $dataToUpdate = [
            'make_and_model_id' => $makeModelId,
            'technical_spec_id' => $techSpecId,
            'asset_type_id'     => $assetTypeId
        ];

        //If manufacturer is apple, remove first 's' from serial number (ticket#MOL-126)
        foreach ($assets as $asset) {

            $dataToUpdate['serial_no'] = $asset->serial_no;

            if (optional($manufacturer)->slug == 'apple' && starts_with(lcfirst($asset->serial_no), 's')) {

                $dataToUpdate['serial_no'] = ltrim(lcfirst($asset->serial_no), 's');
            }

            $existing = Asset::where('serial_no', $dataToUpdate['serial_no'])->where('id', '!=', $asset->id)->first();

            if ($existing) {

                $existing->update(['asset_original_value' => $asset->asset_original_value]);
                AssetHistory::where('asset_id', $asset->id)->delete();
                $asset->delete();

                continue;
            }

            $asset->update($dataToUpdate);
        }
    }
}
