<?php

namespace App\Services\Asn\Presidio;

use App\Services\Asn\AbstractAsnMappingService;
use App\Repositories\Asn\PresidioAssetsRepository;

class AccessoriesMappingService extends AbstractAsnMappingService
{
    /**
     * AccessoriesMappingService Constructor
     *
     * @param PresidioAssetsRepository $presidioRepository
     */
    public function __construct(protected PresidioAssetsRepository $presidioRepository)
    {
        $this->presidioRepository = $presidioRepository;
    }

    /**
     * Retrieve Presidio accessories mapping data
     *
     * Retrieves Presidio accessories mapping data from the database.
     *
     * @return \Illuminate\Database\Eloquent\Collection The Presidio accessories mappings.
     */
    public function getAccessoriesMapping()
    {
        return $this->presidioRepository->getAccessoriesMappingData();
    }

    /**
     * Retrieve Presidio accessories mapping data, as a collection.
     *
     * @return \Illuminate\Database\Eloquent\Collection The Presidio accessories mappings.
     */
    public function getAccessoriesMappingData()
    {
        return $this->getAccessoriesMapping()->get();
    }

    /**
     * Filter and retrieve Presidio accessories mappings.
     *
     * Retrieves accessories mapping data, applying a filter based on the
     * 'searchText' request parameter. If 'searchText' is provided, the function
     * filters the data to include only those entries where the 'part_no' contains
     * the search text. Returns the filtered data set.
     *
     * @return \Illuminate\Database\Eloquent\Collection The filtered accessories mappings.
     */
    public function getAccessoriesMappingSearch()
    {
        $searchText             = request('searchText');
        $presidioMappingData    = $this->getAccessoriesMapping();

        if ($searchText) {
            $presidioMappingData    = $presidioMappingData->where('part_no', 'like', '%' . $searchText . '%');
        }
        return $presidioMappingData->get();
    }

    /**
     * Bulk upload Presidio accessories mappings from a CSV file.
     *
     * The CSV file should be in the format specified in the bulk-upload.asnAccessoriesMappingData config.
     * The function will return a message and an optional errors array.
     * If the file is not a CSV file or does not match the template, the function will return an error message.
     * If the import is successful, the function will return a success message.
     * If there are errors during the import, the function will return an error message and an array of errors.
     *
     * @return array
     */
    public function bulkUpload($request)
    {
        $file       = $request->file('file');
        $extension  = strtolower($file->getClientOriginalExtension());
        $path       = $file->storeAs('public/presidio_mapping_bulk_upload', 'presidio-accessories-mapping-' . date('m-d-y-h-i-s') . '.' . $extension);
        $path       = storage_path('app/' . $path);
        $headerMap  = config('bulk-upload.asnAccessoriesMappingData');

        $checkHeader = $this->checkHeader($path, $headerMap);

        if ($checkHeader) {
            $errors = 'Upload file do not match the template';

            return compact('errors');
        }

        $errors = $this->importAccessoryMappingData($path, 'presidio');

        if ($errors) {
            $message = 'Mapping completed with errors';

            return compact('message', 'errors');
        }

        $message = 'Bulk mapping completed successfully';

        return compact('message');
    }
}
