<?php

namespace App\Services\Asn;

use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class AsnHistoryService
{

    /**
     * To get details from storage/integration folder.
     * 
     * @param string $disk
     * @param string $folderDepth
     * @return array $asnHistoryDetails
     */
    public function getAsnHistoryData($disk = 'integration', $folderDepth = 2)
    {
        $path = storage_path($disk);
        $pathFiles = getLatestFileNames($path);
        $asnHistoryDetails = [];

        if (!is_array($pathFiles) || empty($pathFiles)) {
            return $asnHistoryDetails;
        }

        foreach ($pathFiles as $pathFile) {

            if (!in_array($pathFile, array_keys(config('asn-history.available_suppliers')))) {
                continue;
            }

            $folder = storage_path($disk . '/' . $pathFile);
            $files = getLatestFileNames($folder);

            // Use array_merge to combine the returned array with the main array
            $asnHistoryDetails = array_merge(
                $asnHistoryDetails,
                $this->getAsnHistoryDetails($files, $pathFile, $disk, $folderDepth)
            );
        }

        return $asnHistoryDetails;
    }

    /**
     * Builds the ASN history details from the file list.
     *
     * @param array $files List of files
     * @param string $pathFile Path of the file
     * @param int $folderDepth Depth of the folder
     *
     * @return array ASN history details
     */
    private function getAsnHistoryDetails($files, $pathFile, $disk, $folderDepth)
    {
        $asnHistoryDetails = [];

        if (empty($files)) {
            return $asnHistoryDetails;
        }

        foreach ($files as $file) {

            if (in_array($file, ['..', '.'])) {
                continue;
            }

            $file =   $pathFile . '/' . $file;

            $fileDirectoryStructure =  Str::of($file)->explode('/');

            //for avoiding nested folder structure like /cdw/any-other-folder/file.csv we want only /cdw/file.csv
            if (count($fileDirectoryStructure) == $folderDepth) {

                $asnHistoryDetails[] = $this->buildSingleAsnRecord($disk, $file, $fileDirectoryStructure);
            }
        }

        return $asnHistoryDetails;
    }

    /**
     * Build a single ASN (Advanced Shipping Notice) record.
     *
     * @param string                         $disk                   The disk where the file is located.
     * @param string                         $file                   The file path.
     * @param \Illuminate\Support\Collection $fileDirectoryStructure The directory structure of the file.
     *
     * @return array The detailed ASN record.
     */
    public function buildSingleAsnRecord($disk, $file, $fileDirectoryStructure): array
    {
        $timestamp = Storage::disk($disk)->lastModified($file);
        $filePath = $disk . '/' . $file;
        $date = timestampToDesiredFormat($timestamp);

        return [
            'supplier' => $fileDirectoryStructure[0],
            'supplier_display_name' => config('asn-history.available_suppliers.' . $fileDirectoryStructure[0], $fileDirectoryStructure[0]),
            'filename' => $fileDirectoryStructure[1],
            'date_added' => $date,
            'timestamp' => $timestamp,
            'download' => sprintf(
                '<a href="%s">Download</a>',
                route('asn-history.download-file', ['file_path' => $filePath]),
            ),
            'sync' => sprintf(
                '<a href="#" class="resync" data-supplier="%s" data-filename="%s">Resync</a>',
                $fileDirectoryStructure[0],
                $fileDirectoryStructure[1],
            ),
        ];
    }

    /**
     * Re-index ASN (Advanced Shipping Notice) data.
     *
     * @param array $asnData The ASN data to be re-indexed.
     *
     * @return array The re-indexed ASN data.
     */
    public function reIndexAsnData($asnData): array
    {
        $asnData = collect($asnData)->sortByDesc('timestamp')->toArray();
        $i = 1;

        foreach ($asnData as $key => $data) {
            $asnData[$key]['id'] = $i++;
        }

        return $asnData;
    }

    /**
     * Get paginated data for DataTables.
     *
     * @param array                    $asnData The ASN data to be paginated.
     * @param \Illuminate\Http\Request $request The request object containing pagination parameters.
     *
     * @return array The paginated ASN data.
     */
    public function getDataForDataTable($asnData, $request)
    {
        $asnData = collect($asnData);
        $start = $request->get('start', 0);
        $perPage = $request->get('length', 50);
        $startPage = ($start / $perPage) + 1;

        $result = $asnData->forPage($startPage, $perPage);

        return $result->values()->toArray();
    }

    /**
     * Get filtered ASN (Advanced Shipping Notice) data.
     *
     * @param array                    $asnData The ASN data to be filtered.
     * @param \Illuminate\Http\Request $request The request object containing filter parameters.
     *
     * @return array The filtered ASN data.
     */
    public function getFilteredData($asnData, $request): array
    {
        $asnData = $this->filterBySupplier($asnData, $request->supplier);
        $asnData = $this->filterByDate($asnData, $request->date_from, $request->date_to);

        return $asnData;
    }

    /**
     * Filter ASN (Advanced Shipping Notice) data by supplier.
     *
     * @param array       $asnData  The ASN data to be filtered.
     * @param string|null $supplier The supplier to filter by.
     *
     * @return array The filtered ASN data.
     */
    public function filterBySupplier($asnData, $supplier): array
    {
        if (!$supplier) {
            return $asnData;
        }
        $asnData = Arr::where($asnData, function ($value) use ($supplier) {
            $supplier = Str::lower($supplier);
            $value = Str::lower($value['supplier']);

            return $value == $supplier;
        });

        return $asnData;
    }

    /**
     * Filter ASN (Advanced Shipping Notice) data by date range.
     *
     * @param array       $asnData  The ASN data to be filtered.
     * @param string|null $dateFrom The start date of the range.
     * @param string|null $dateTo   The end date of the range.
     *
     * @return array The filtered ASN data.
     */
    public function filterByDate($asnData, $dateFrom, $dateTo): array
    {
        if (!$dateFrom || !$dateTo) {
            return $asnData;
        }

        $asnData = Arr::where($asnData, function ($value) use ($dateFrom, $dateTo) {
            $dateFrom = Carbon::createFromFormat(config('asn-history.date_format'), $dateFrom);
            $dateTo = Carbon::createFromFormat(config('asn-history.date_format'), $dateTo);

            if ($value['timestamp'] >= $dateFrom->getTimestamp() && $value['timestamp'] <= $dateTo->getTimestamp()) {
                return true;
            }
        });

        return $asnData;
    }

    /**
     * Get the resync command for a specific supplier and file.
     *
     * @param \Illuminate\Http\Request $request The request object containing the supplier and file name.
     *
     * @return string The constructed resync command or an empty string if the command is not found.
     */
    public function getResyncCommand($request): string
    {
        $supplier = Str::lower($request->get('supplier', ''));
        $command = config('asn-history.' . $supplier . '.signature');

        if (!$command) {
            return '';
        }

        $fileName = Str::replace('"', '\"', $request->get('file_name', ''));

        return $command . ' "' . $fileName . '"';
    }
}
