<?php

namespace App\Services\Asn;

use Facades\App\Repositories\Asn\AsnAssetsRepository;
use App\Models\AsnHardwareMapping;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

abstract class AbstractAsnAssetsService
{

    abstract protected function getRepository();

    public function __construct()
    {
    }

    /**
     * Retrieve and process asset data based on the given status slug.
     *
     * @param string $statusSlug The status slug used to filter assets.
     *
     * @return array An array containing the filtered assets and their count.
     */
    public function data($statusSlug)
    {
        $inputData  = $this->getInputData();
        $assets     = $this->getRepository()::getAssets($statusSlug);
        $assets     = $this->filter($assets, $inputData);

        $count      = $assets->count();
        $assets     = $this->getOutputData($assets);

        return compact('assets', 'count');
    }

    /**
     * Retrieve and process input data from the request form.
     *
     * @return array An associative array containing the input data.
     */
    public function getInputData()
    {
        return [
            'searchText'      => isset(request('form')['searchText']) ? request('form')['searchText'] : '',
            'po_no'           => isset(request('form')['po_no']) ? request('form')['po_no'] : '',
            'serial_no'       => isset(request('form')['serial_no']) ? request('form')['serial_no'] : '',
            'asset_tag'       => isset(request('form')['asset_tag']) ? request('form')['asset_tag'] : '',
            'shipment_status' => isset(request('form')['shipment_status']) ? request('form')['shipment_status'] : '',
        ];
    }

    /**
     * Filter the assets based on the provided input data.
     *
     * @param \Illuminate\Database\Eloquent\Collection $assets    The collection of assets to be filtered.
     * @param array                                    $inputData The input data used for filtering the assets.
     *
     * @return \Illuminate\Database\Eloquent\Collection The filtered collection of assets.
     */
    public function filter($assets, $inputData)
    {
        $assets = $assets->RelationData($inputData['po_no'], 'po_id');
        $assets = $assets->RelationData($inputData['serial_no'], 'serial_no');
        $assets = $assets->RelationData($inputData['asset_tag'], 'asset_tag');
        $assets = $this->getRepository()::filterWithTracking($assets, $inputData['shipment_status'], 'shipment_status');
        $assets = $this->getRepository()::filterByPo($assets, $inputData['searchText']);

        return $assets;
    }

    /**
     * Retrieve and process the output data for the users with pagination and sorting.
     *
     * @param \Illuminate\Database\Eloquent\Builder $users The query builder instance for the users.
     *
     * @return \Illuminate\Database\Eloquent\Collection The collection of users after applying pagination and sorting.
     */
    public function getOutputData($users)
    {
        $start = request('start');
        $limit = request('length');

        if ($limit != -1) {
            $users = $users->offset($start)
                ->limit($limit);
        }

        $users->orderBy('id', 'DESC');

        return $users->get();
    }

    /**
     * Retrieve and process asset data for export based on the given status slug.
     *
     * @param string $statusSlug The status slug used to filter assets.
     *
     * @return \Illuminate\Database\Eloquent\Collection The filtered collection of assets for export.
     */
    public function getExportData($statusSlug)
    {
        $inputData = $this->getExportInputData();
        $assets    = $this->getRepository()::getAssets($statusSlug);
        $assets    = $this->filter($assets, $inputData);

        return $assets;
    }

    /**
     * Retrieve and process input data for export from the request.
     *
     * @return array An associative array containing the input data for export.
     */
    public function getExportInputData()
    {
        return [
            'po_no'         => request('po_no') ?? request('po_no'),
            'serial_no'     => request('serial_no') ?? request('serial_no'),
            'asset_tag'     => request('asset_tag') ?? request('asset_tag'),
            'shipment_status' => request('shipment_status') ?? request('shipment_status'),
            'searchText'     => isset(request('form')['searchText']) ? request('form')['searchText'] : '',
        ];
    }

    /**
     * Process and retrieve ASN asset data with nested data structure.
     *
     * @param \Illuminate\Database\Eloquent\Collection $assets The collection of assets to process.
     * @param int                                      $start  The starting index for processing assets.
     * @param array                                    $data   The existing data array to append the processed assets.
     *
     * @return array The updated data array with processed assets.
     */
    public function getAsnAssetData($assets, $start, $data)
    {
        $parentIndex = $start;

        foreach ($assets as $asset) {
            $parentIndex++;

            $name = route('assets.show', $asset->id);

            $nestedData = $this->getNestedData($asset, $parentIndex, $name);

            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieve nested data for an asset.
     *
     * @param Asset  $asset The asset object to retrieve nested data from.
     * @param int    $index The index used for identifying the asset.
     * @param string $route The route URL for the asset's detailed view.
     *
     * @return array An associative array containing the nested data for the asset.
     */
    public function getNestedData($asset, $index, $route)
    {
        $nestedData['id']           = $index;
        $nestedData['asset_tag']    = "<a href=$route>" . $asset->asset_tag . "</a>";
        $nestedData['serial_no']    = "<a href=$route>" . $asset->serial_no . "</a>";
        $nestedData['po_no']        = $asset->po_id;
        $nestedData['description']  = $asset->description;
        $nestedData['part_no']  = $asset->part_no;
        $nestedData['hardware']     = $asset->makeAndModel ? $asset->makeAndModel->name : '';
        $nestedData['asset_type']   = $asset->assetType ? $asset->assetType->name : '';
        $nestedData['tech_spec']    = $asset->technicalSpec ? $asset->technicalSpec->details : '';
        $nestedData['tracking_number'] = $asset->assetTracking ? $asset->assetTracking->tracking_number : '';
        $nestedData['shipment_status'] = $asset->assetTracking ? $asset->assetTracking->shipment_status : '';
        $nestedData['checkbox'] = $this->getReceiveAssetCheckbox($asset);
        $nestedData['edit']   = $this->getActionButtons($asset);

        return $nestedData;
    }

    /**
     * Generate a checkbox HTML element for receiving an asset.
     *
     * @param \App\Models\Asset $asset The asset object for which the checkbox is generated.
     *
     * @return string The HTML string for the checkbox element.
     */
    public function getReceiveAssetCheckbox($asset)
    {
        $checkbox = '<input type="checkbox" class="receiveAsset checkSingle"   name="receiveAsset" value="' . $asset->id . '"';

        if ((!$asset->makeAndModel) || (!$asset->assetType) || (!$asset->technicalSpec) || ($asset->serial_no == '')  || ($asset->po_id == '')) {
            $checkbox .= 'disabled';
        }

        $checkbox .= ' >';

        return $checkbox;
    }

    /**
     * Generate action buttons HTML for an asset.
     *
     * @param \App\Models\Asset $asset The asset object for which the action buttons are generated.
     *
     * @return string The HTML string for the action buttons.
     */
    public function getActionButtons($asset)
    {
        $buttonMode = " data-toggle='modal' data-target='#receiveAssetModal' ";

        if ((!$asset->makeAndModel) || (!$asset->assetType) || (!$asset->technicalSpec) || ($asset->serial_no == '') || ($asset->asset_tag == '') || ($asset->po_id == '')) {
            $buttonMode = "disabled";
        }

        $actions = "<div style='width:175px'> <a  class='btn recieve-asn-asset btn-sm btn-info' style='margin-right:4px;' data-asset='" . $asset->id . "'  $buttonMode >Receive</a>";

        $actions .= "<a href='' class='btn btn-primary btn-sm'  data-toggle='modal' data-asset-id='" . $asset->id . "' data-target='#editModal'><i class='icon icon-n-edit' aria-hidden='true'  ></i></a>";

        $actions .= " <a  class='btn btn-danger btn-sm delete-asn-asset' data-id='" . $asset->id . "' data-toggle='modal' data-target='#deleteAssetModal'><i class='icon icon-delete-forever'></i></a> ";

        // $asnHardwareMapping = AsnHardwareMapping::where('description', $asset->description)->first();
        $asnHardwareMapping = AsnHardwareMapping::where('part_no', $asset->part_no)->orWhere('part_no', $asset->description)->first();

        if ($asnHardwareMapping == null) {
            $actions  .= " <a  class='btn btn-info btn-sm' data-description='" . disableCSVInjection($asset->description) . " data-toggle='modal' data-target='#addAsnMapping' onclick='addAsnMapping(`" . $asset->description . "`)' data-toggle='tooltip' data-placement='right' title='Add Mapping'><i class='icon icon-e-add'></i></a>";
        }

        return $actions  .= "</div>";
    }

    /**
     * Retrieve and process ASN asset export data with nested data structure.
     *
     * @param \Illuminate\Database\Eloquent\Collection $assets  The collection of assets to process.
     * @param int                                      $start   The starting index for processing assets.
     * @param array                                    $data    The existing data array to append the processed assets.
     * @param string|null                              $asnName The ASN name, if available, to include in the nested data.
     *
     * @return array The updated data array with processed assets.
     */
    public function getAsnAssetExportData($assets, $start, $data, $asnName = null)
    {
        $parentIndex = $start;

        foreach ($assets as $asset) {
            $parentIndex++;
            $nestedData = $this->getExportNestedData($asset, $asnName);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieve nested data for an asset for export purposes.
     *
     * @param Asset       $asset   The asset object to retrieve nested data from.
     * @param string|null $asnName The ASN name to include in the nested data, if available.
     *
     * @return array An associative array containing the nested data for the asset.
     */
    public function getExportNestedData($asset, $asnName)
    {
        $nestedData["Asset Tag # (" . $asnName . ")"] = disableCSVInjection($asset->asset_tag);
        $nestedData["Serial # (" . $asnName . ")"]  = disableCSVInjection($asset->serial_no);
        $nestedData["PO # (" . $asnName . ")"]      = disableCSVInjection($asset->po_id);
        $nestedData["Description (" . $asnName . ")"] = disableCSVInjection($asset->part_no);
        $nestedData["Hardware Standard (Teqtivity)"] = $asset->makeAndModel ? disableCSVInjection($asset->makeAndModel->name) : '';
        $nestedData["Tech Specs (Teqtivity)"] = $asset->technicalSpec ? disableCSVInjection($asset->technicalSpec->details) : '';
        $nestedData["Tracking Number (" . $asnName . ")"] = $asset->assetTracking ? disableCSVInjection($asset->assetTracking->tracking_number) : '';
        $nestedData["Shipment Status"] = $asset->assetTracking ? disableCSVInjection($asset->assetTracking->shipment_status) : '';

        return $nestedData;
    }

    /**
     * Validate the asset data from the request.
     *
     * @return \Illuminate\Support\MessageBag|bool Returns validation errors if validation fails, otherwise returns false.
     */
    public function validateAsset()
    {
        $validator = Validator::make(
            request()->all(),
            [
                'id'                 => 'required',
                'asset_tag'          => ['nullable', Rule::unique('assets')->ignore(request('id'))],
                'serial_no'          => ['required', Rule::unique('assets')->ignore(request('id'))],
                'po_id'              => 'required',
                'asset_type_id'      => 'required',
                'make_and_model_id'  => 'required',
            ],
            $messages = [
                'make_and_model_id.required'   => 'Hardware Standard is required.',
                'id.required'                  => 'Something went wrong. Try again',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }

        return false;
    }
}
