<?php

namespace App\Services\Api;

use Illuminate\Support\Facades\Validator;
use App\Models\Department;
use App\Models\EmployeePosition;
use App\Models\EmployeeType;
use App\Models\UserType;
use App\Rules\UniqueEmployeeIdForEmail;
use App\User;

/**
 * Class UserUpdateApiService
 *
 * Service class for creating and validating user data from API requests.
 */
class UserUpdateApiService extends ApiService
{
    /**
     * Get and prepare user data from the request.
     *
     * @return array|bool User data array or false if validation fails.
     */
    public function getData()
    {
        $inputData = request()->all();

        $userData = $this->initializeUserData($inputData);


        $relationalValues = ['department', 'title', 'employee_type'];

        $this->populateDepartmentData($inputData, $userData);
        $this->populatePositionData($inputData, $userData);
        $this->populateEmployeeTypeData($inputData, $userData);

        $hasRelationalValues = array_intersect_key(array_flip($relationalValues), $inputData);
        if (empty($hasRelationalValues)) {
            return $userData;
        }

        return $userData;
    }

    /**
     * Validate the fields from the request.
     *
     * @return \Illuminate\Contracts\Validation\Validator|null Validation errors or null if validation succeeds.
     */
    public function validateFields()
    {
        $rules = [
            'first_name' => 'nullable',
            'last_name' => 'nullable',
            'email' => 'required|exists:users',
            'employee_id' => ['nullable', new UniqueEmployeeIdForEmail(request('email'))],
            'hire_date'         => 'nullable|date_format:m/d/Y',
            'terminated_date'  => 'nullable|date_format:m/d/Y',
            'legalhold' => 'nullable|in:Yes,No',
            'status' => 'nullable|in:Active,Terminated'
        ];

        $messages = [
            'first_name.required' => 'The First Name is required.',
            'last_name.required'     => 'The Last Name is required.',
            'email.required'     => 'The Email is required.',
            'hire_date.required'     => 'The Hire Date is required.',
            'employee_id.required'     => 'The Employee ID is required.',
            'email.unique'       => 'The Email has already been taken.',
            'employee_id.unique'       => 'The Employee ID has already been taken.',
            'legalhold.in' => 'The legalhold field must be either Yes or No.',
            'status.in' => 'The status field must be either Active or Terminated.',
        ];

        $validator = Validator::make(request()->all(), $rules, $messages);

        return $validator->fails() ? $validator->errors() : null;
    }

    /**
     * Save location data to the database.
     *
     * @param array $userData Location data to be saved.
     *
     * @return void
     */
    public function saveData($userData)
    {
        $user = User::where('email', $userData['email'])->first();

        if ($user) {
            $user->update($userData);

            if (!$user->hasRole($user->userType->name)) {
                $user->syncRoles([$user->userType->name]);
            }
        }
    }

    /**
     * Initialize location data from the input data.
     *
     * @param array $inputData Input data.
     *
     * @return array Initialized location data.
     */
    private function initializeUserData($inputData)
    {
        $directValues = [
            'first_name', 'last_name', 'employee_id', 'hire_date', 'email', 'work_location',
            'terminated_date', 'legalhold', 'status'
        ];

        foreach ($inputData as $key => $value) {
            if (in_array($key, $directValues)) {
                $userData[$key] = $value;
            }
        }

        if (array_key_exists('work_location', $userData)) {
            $userData = array_merge($userData, ['city' => $userData['work_location'], 'location' => $userData['work_location']]);
            unset($userData['work_location']);
        }

        if (isset($userData['hire_date']) && !empty($userData['hire_date'])) {
            $userData['hire_date'] = convert_to_db_datetime($userData['hire_date']);
        }

        if (isset($userData['terminated_date']) && !empty($userData['terminated_date'])) {
            $userData['terminated_date'] = convert_to_db_datetime($userData['terminated_date']);
            //$userData['status'] = 0;
        }

        if (isset($userData['legalhold']) && !empty($userData['legalhold'])) {
            $userData['legalhold'] = (strtolower($userData['legalhold']) == 'yes') ? 1 : 0;
        }

        if (isset($userData['status']) && !empty($userData['status'])) {
            $userData['status'] = (strtolower($userData['status']) == 'active') ? 1 : 0;
        }

        return $userData;
    }

    /**
     * Populate department data in the user data array.
     *
     * @param array $inputData
     * @param array $userData
     *
     * @return void
     */
    private function populateDepartmentData($inputData, &$userData)
    {
        if (isset($inputData['department']) && !empty($inputData['department'])) {
            $slug = str_slug($inputData['department'], '_');
            $department = Department::firstOrCreate(['slug' => trim($slug)], ['name' => $inputData['department']]);
            $userData['department_id'] = $department->id;
        }
    }

    /**
     * Populate position data in the user data array.
     *
     * @param array $inputData
     * @param array $userData
     *
     * @return void
     */
    private function populatePositionData($inputData, &$userData)
    {
        if (isset($inputData['title']) && !empty($inputData['title'])) {
            $slug = str_slug($inputData['title'], '_');

            $position = EmployeePosition::firstOrCreate(
                ['slug' => trim($slug)],
                ['name' => $inputData['title']]
            );

            $userData['position_id'] = $position->id;
        }
    }

    /**
     * Populate employee type data in the user data array.
     *
     * @param array $inputData
     * @param array $userData
     *
     * @return void
     */
    private function populateEmployeeTypeData($inputData, &$userData)
    {
        if (isset($inputData['user_type']) && !empty($inputData['user_type'])) {
            $slug = str_slug($inputData['user_type'], '_');

            $employeeType = EmployeeType::firstOrCreate(
                ['slug' => trim($slug)],
                ['name' => $inputData['user_type']]
            );

            $userData['employee_type_id'] = $employeeType->id;
        }
    }
}
