<?php

namespace App\Services\Api;

use App\Http\Requests\Api\GetUserRequest;
use App\Repositories\Api\UserApiRepository;
use App\Http\Resources\User\UserCollection;
use Illuminate\Http\Request;

/**
 * Service class for User API.
 */
class UserApiService extends ApiService
{
    /**
     * @var UserApiRepository
     */
    protected $userApiRepository;

    /**
     * Create a new UserApiService instance.ATBBy4BW4B5s6p3Y5X4W26YMpE2g030370A6ATBBy4BW4B5s6p3Y5X4W26YMpE2g030370A6
     *
     * @param UserApiRepository $userApiRepository
     */
    public function __construct(UserApiRepository $userApiRepository)
    {
        $this->userApiRepository = $userApiRepository;
    }

    /**
     * Get all users with pagination.
     *
     * @param GetUserRequest $request The HTTP request object.
     *
     * @return UserCollection
     */
    public function getUsers(GetUserRequest $request)
    {
        $users = $this->userApiRepository->getUsers();
        $users = $this->filterUsers($request, $users);
        $limit     = $this->getQueryLimit($request->limit);

        return new UserCollection($users->paginate($limit));
    }


    /**
     * Search users.
     *
     * @param Request $request The HTTP request object.
     *
     * @return UserCollection
     */
    public function searchUsers(Request $request)
    {
        $users = $this->userApiRepository->getUsers();
        $users = $this->userApiRepository->searchUsersWithGeneralQueries($users, $request->search);

        $limit     = $this->getQueryLimit($request->limit);

        return new UserCollection($users->paginate($limit));
    }

    /**
     * Filter users based on inputs.
     *
     * @param GetUserRequest $request The HTTP request object.
     * @param mixed          $users   The users to filter.
     *
     * @return mixed
     */
    public function filterUsers(GetUserRequest $request, $users)
    {
        foreach (config('api-filter-fields.users.filter_fields') as $key) {
            if ($request->has($key)) {
                $users = $this->userApiRepository->filterByDirectFields($users, $request->$key, $key);
            }
        }

        foreach (config('api-filter-fields.users.filter_mapping_fields') as $key => $value) {
            if ($request->has($key)) {
                $users = $this->userApiRepository->filterByDirectFields($users, $request->$key, $value);
            }
        }

        foreach (config('api-filter-fields.users.filter_fields_bool') as $key => $value) {
            if ($request->has($key)) {
                $requestArray = array_map('strtolower', $request->$key);
                $users = $this->userApiRepository->filterByDirectFields($users, array_values(array_intersect_key($value, array_flip($requestArray))), $key);
            }
        }

        foreach (config('api-filter-fields.users.relational_filter_fields') as $key => $value) {
            if ($request->has($key)) {
                $users = $this->userApiRepository->filterByRelatedModelFields($users, $request->$key, $value['relation'], $value['db_field']);
            }
        }

        foreach (config('api-filter-fields.users.filter_date_fields') as $key => $value) {
            if ($request->has($key)) {
                $users = $this->userApiRepository->filterByDateFields($users, $request->$key, $value['db_field']);
            }
        }

        return $users;
    }
}
