<?php

namespace App\Services\Api;

use App\Http\Requests\Api\GetTechSpecRequest;
use App\Http\Resources\TechSpec\TechSpecCollection;
use Illuminate\Http\Request;
use App\Repositories\Api\TechSpecApiRepository;

/**
 * Service Class for Techspec API
 */
class TechSpecApiService extends ApiService
{
	/**
     * @var TechSpecApiRepository
     */
    protected $techSpecApiRepository;

	/**
     * Create a new TechSpecApiService instance.
     *
     * @param TechSpecApiRepository $techSpecApiRepository
     */
	public function __construct(TechSpecApiRepository $techSpecApiRepository)
	{
		$this->techSpecApiRepository = $techSpecApiRepository;
	}

	/**
     * Get the techSpecs.
     *
     * @param GetTechSpecRequest $request The HTTP request object.
     *
     * @return TechSpecCollection
     */
	public function getTechSpecs(GetTechSpecRequest $request)
	{
		$techSpecs = $this->techSpecApiRepository->getTechSpecs();
		$techSpecs = $this->filterTechSpecs($request, $techSpecs);
		$limit     = $this->getQueryLimit($request->limit);

		return new TechSpecCollection($techSpecs->paginate($limit));
	}

	/**
     * Filter techSpecs based on inputs.
     *
     * @param GetTechSpecRequest $request   The HTTP request object.
     * @param mixed              $techSpecs The tech specs to filter.
     *
     * @return mixed
     */
	public function filterTechSpecs(GetTechSpecRequest $request, $techSpecs)
	{
		foreach (config('api-filter-fields.techSpecs.filter_fields') as $key) {
			if ($request->has($key)) {
				$techSpecs = $this->techSpecApiRepository->filterByDirectFields($techSpecs, $request->$key, $key);
			}
		}

		foreach (config('api-filter-fields.techSpecs.relational_filter_fields') as $key => $value) {
			if ($request->has($key)) {
				$techSpecs = $this->techSpecApiRepository->filterByRelatedModelFields($techSpecs, $request->$key, $value['relation'], $value['db_field']);
			}
		}

		foreach (config('api-filter-fields.techSpecs.filter_date_fields') as $key => $value) {
			if ($request->has($key)) {
				$techSpecs = $this->techSpecApiRepository->filterByDateFields($techSpecs, $request->$key, $value['db_field']);
			}
		}

		return $techSpecs;
	}

	/**
     * Search techSpecs.
     *
     * @param Request $request The HTTP request object.
     *
     * @return TechSpecCollection
     */
	public function searchTechSpecs(Request $request)
	{
		$techSpecs = $this->techSpecApiRepository->getTechSpecs();
		$techSpecs = $this->techSpecApiRepository->searchTechSpecsWithGeneralQueries($techSpecs, $request->search);

		$limit     = $this->getQueryLimit($request->limit);

		return new TechSpecCollection($techSpecs->paginate($limit));
	}
}
