<?php 

namespace App\Services\Api\Slackbot;

use App\Services\Api\Jira\TeqtivityAssetStatus;
use App\Services\Api\ApiService;
use Illuminate\Http\Request;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\Asset;
use Exception;
use App\User;
use Cache;
use Auth;
use Log;

/**
 * Service class for assign asset slash command
 */
class AssignAssetService extends SlackbotApiService
{
    public function __construct(protected TeqtivityAssetStatus $statusService, protected ApiService $responseService)
	{
	}

    /**
     * Get Asset and user details for assign asset
     *
     * @param Request $request
     *
     * @return Collection
     */
    public function getAssetDetails(Request $request)
    {

        $user = $this->getUserByEmail($request->user_email);

        if (isset($user['error']) && $user['error'] == true) {
            return $user;
        }

        //check permission
        if (!$user->can('Slackbot Admin Tools')) {
            return $this->getErrorResponse('You do not have permission to update this asset');
        }

        //Log::info('Searched serial'. $request->serial_no);
        //Check the asset exists
        $asset = Asset::select('id', 'serial_no', 'asset_tag', 'asset_status_id')->with('assetStatus:id,name,slug')
                    ->where(function($query) use ($request){
                        $query->where('asset_tag', $request->serial_no)
                              ->orWhere('serial_no', $request->serial_no);
                    })
                    ->first();

        if (!$asset) {
            return $this->getErrorResponse('Asset not found in Teqtivity');
        }

        //Check the assign Status is existing on the Teqtivity
        $assignStatus = AssetStatus::select('id', 'name', 'slug')->where('slug', config('slackbot.asset_status.assigned'))->first();
        if (!$assignStatus) {
            return $this->getErrorResponse('Assign Status not found in Teqtivity');
        }

        //Check the asset is valid for the status update
        $currentStatusSlug = optional($asset->assetStatus)->slug;
        if (!in_array(config('slackbot.asset_status.assigned'), config('asset-status.' . $currentStatusSlug))) {
            return $this->getErrorResponse('The asset cannot be assigned because of status restrictions.');
        }

        //Check the assignee user exists
        $assigneeUser = $this->getUserByEmail($request->assignee_email);
        if (isset($assigneeUser['error']) && $assigneeUser['error'] == true) {
            return $user;
        }

        //Inject the assignee user and assign status to the asset collection
        $asset->assignee = $assigneeUser;
        $asset->assign_status = $assignStatus;
        //Log::info('Fetched asset'. $asset);
        return $this->getSuccessReponse($asset, 'Asset');
    }

    /**
     * Update the asset status.
     *
     * @param Request $request 
     */
    public function updateAssetStatus(Request $request)
    {
        $error = $this->statusService->validate($request);
		if ($error) {
			return $this->responseService->errorResponse($error);
		}
		try {
			$response = $this->statusService->update($request);

			return $this->responseService->successResponse('', $response);
		} catch (Exception $e) {
			Log::channel('single')->info('Slackbot Status Update Error : ' . $e->getMessage());
			return $this->responseService->errorResponse('Something went wrong. Try again later');
		}
    }
}