<?php

namespace App\Services\Api;

use App\Http\Requests\Api\GetLocationRequest;
use App\Repositories\Api\LocationApiRepository;
use App\Http\Resources\Location\LocationCollection;
use Illuminate\Http\Request;

/**
 * Service class for Location API
 */
class LocationApiService extends ApiService
{
    protected $locationApiRepository;

    /**
     * Create a new instance of LocationApiService.
     *
     * @param LocationApiRepository $locationApiRepository
     */
    public function __construct(LocationApiRepository $locationApiRepository)
    {
        $this->locationApiRepository = $locationApiRepository;
    }

    /**
     * Get all locations with pagination.
     *
     * @param GetLocationRequest $request
     *
     * @return LocationCollection
     */
    public function getLocations(GetLocationRequest $request)
    {
        $locations = $this->locationApiRepository->getLocations();
        $locations = $this->filterLocations($request, $locations);
        $limit     = $this->getQueryLimit($request->limit);

        return new LocationCollection($locations->paginate($limit));
    }

    /**
     * Filter locations based on inputs.
     *
     * @param GetLocationRequest $request
     * @param mixed $locations
     *
     * @return mixed
     */
    public function filterLocations(GetLocationRequest $request, $locations)
    {
        foreach (config('api-filter-fields.locations.filter_fields') as $key) {
            if ($request->has($key)) {
                $locations = $this->locationApiRepository->filterByDirectFields($locations, $request->$key, $key);
            }
        }

        foreach (config('api-filter-fields.locations.relational_filter_fields') as $key => $value) {
            if ($request->has($key)) {
                $locations = $this->locationApiRepository->filterByRelatedModelFields($locations, $request->$key, $value['relation'], $value['db_field']);
            }
        }

        foreach (config('api-filter-fields.locations.filter_date_fields') as $key => $value) {
            if ($request->has($key)) {
                $locations = $this->locationApiRepository->filterByDateFields($locations, $request->$key, $value['db_field']);
            }
        }

        return $locations;
    }

    /**
     * Search locations.
     *
     * @param Request $request
     *
     * @return LocationCollection
     */
    public function searchLocations(Request $request)
    {
        $locations = $this->locationApiRepository->getLocations();
        $locations = $this->locationApiRepository->searchLocationsWithGeneralQueries($locations, $request->search);

        $limit     = $this->getQueryLimit($request->limit);

        return new LocationCollection($locations->paginate($limit));
    }
}
