<?php

namespace App\Services\Api\Jira;

use App\Models\AssetStatus;
use App\Models\Asset;
use App\User;
use Cache;

class ReporterAssets
{

    /**
     * Get the assets associated with a user by email.
     *
     * @param string $email The email of the user.
     *
     * @return array The response containing user and assets data or an error message.
     */
    public function getAssets($email)
    {
        $user = User::select('id', 'first_name', 'last_name', 'email')
            ->where('email', $email)
            ->first();

        if (!$user) {
            return [
                'error'         => true,
                'message'       => 'User not found in Teqtivity',
                'status_code'   => 404,
            ];
        }

        $assets = Asset::with('assetStatus', 'assetType', 'makeAndModel', 'technicalSpec')->where('user_id', $user->id)->get();

        $assets = $this->getAvailableStatusesForUpdate($assets);

        $data['user']   = $user;
        $data['assets'] = $assets;

        return [
            'success'     => true,
            'message'     => 'Assets',
            'status_code' => 200,
            'data'        => $data,
        ];
    }

    /**
     * Get the available statuses for updating each asset.
     *
     * @param \Illuminate\Database\Eloquent\Collection $assets The collection of assets.
     *
     * @return \Illuminate\Database\Eloquent\Collection The collection of assets with available statuses for update.
     */
    public function getAvailableStatusesForUpdate($assets)
    {
        foreach ($assets as $asset) {
            $allowedStatusForAsset = config('asset-status.' . $asset->assetStatus->slug);
            $statuses = Cache::remember('asset-statuses', 43200, function () {
                return AssetStatus::get();
            });

            $asset['available_statuses'] = $statuses->filter(function ($status) use ($allowedStatusForAsset) {
                return in_array($status->slug, $allowedStatusForAsset);
            });
        }

        return $assets;
    }
}
