<?php

namespace App\Services\Api;

use App\Http\Requests\Api\GetHardwareRequest;
use App\Http\Resources\Hardware\HardwareCollection;
use App\Repositories\Api\HardwareStandardApiRepository;
use Illuminate\Http\Request;

/**
 * Service class for hardware list API
 */
class HardwareStandardApiService extends ApiService
{
    /**
     * @var HardwareStandardApiRepository
     */
    protected $hardwareStandardApiRepository;

    /**
     * Create a new HardwareStandardApiService instance.
     *
     * @param HardwareStandardApiRepository $hardwareStandardApiRepository
     */
    public function __construct(HardwareStandardApiRepository $hardwareStandardApiRepository)
    {
        $this->hardwareStandardApiRepository = $hardwareStandardApiRepository;
    }

    /**
     * Get all hardware standards.
     *
     * @param GetHardwareRequest $request The HTTP request.
     *
     * @return HardwareCollection
     */
    public function getHardwares(GetHardwareRequest $request)
    {
        $hardwareStandards = $this->hardwareStandardApiRepository->getHardwareStandards();
        $hardwareStandards = $this->filterHardwareStandards($request, $hardwareStandards);
        $limit     = $this->getQueryLimit($request->limit);

        return new HardwareCollection($hardwareStandards->paginate($limit));
    }

     /**
     * Filter hardwareStandards based on inputs.
     *
     * @param GetHardwareRequest $request
     * @param mixed $hardwareStandards
     *
     * @return mixed
     */
    public function filterHardwareStandards(GetHardwareRequest $request, $hardwareStandards)
    {

        foreach (config('api-filter-fields.hardwareStandards.filter_fields') as $key) {
            if ($request->has($key)) {
                $hardwareStandards = $this->hardwareStandardApiRepository->filterByDirectFields($hardwareStandards, $request->$key, $key);
            }
        }

        foreach (config('api-filter-fields.hardwareStandards.relational_filter_fields') as $key => $value) {
            if ($request->has($key)) {
                $hardwareStandards = $this->hardwareStandardApiRepository->filterByRelatedModelFields($hardwareStandards, $request->$key, $value['relation'], $value['db_field']);
            }
        }

        foreach (config('api-filter-fields.hardwareStandards.filter_date_fields') as $key => $value) {
            if ($request->has($key)) {
                $hardwareStandards = $this->hardwareStandardApiRepository->filterByDateFields($hardwareStandards, $request->$key, $value['db_field']);
            }
        }

        return $hardwareStandards;
    }

    /**
     * Search hardwareStandards.
     *
     * @param Request $request The HTTP request.
     *
     * @return HardwareCollection
     */
    public function searchHardwareStandards(Request $request)
    {
        $hardwareStandards = $this->hardwareStandardApiRepository->getHardwareStandards();
        $hardwareStandards = $this->hardwareStandardApiRepository->searchHardwareStandardsWithGeneralQueries($hardwareStandards, $request->search);

        $limit     = $this->getQueryLimit($request->limit);

        return new HardwareCollection($hardwareStandards->paginate($limit));
    }
}
