<?php

namespace App\Services\Api;

use App\Models\Asset;
use App\Models\AssetType;
use App\Models\Carrier;
use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;
use Validator;
use Carbon\Carbon;
use App\Events\BulkUpdates;
use App\Rules\AssetTypeValidation;
use App\Rules\UniqueAssetTagForSerialNo;

/**
 * Service class for Asset Update API.
 */
class AssetUpdateApiService extends ApiService
{
    /**
     * Get data for asset update.
     *
     * @return array|false
     */
    public function getData()
    {
        $inputData = request()->all();

        $directValues = ['asset_tag', 'po_id', 'serial_no', 'carrier_id', 'imei', 'created_at', 'warranty_end_date', 'lease_start_date', 'lease_end_date', 'asset_original_value', 'lost_date', 'end_of_life_date', 'lock_status', 'unlock_code', 'lock_notes'];
        $relationalValues = ['asset_type', 'hardware_standard', 'technical_spec'];

        foreach ($inputData as $key => $value) {
            if (in_array($key, $directValues)) {
                $assetData[$key] = $value;
            }
        }

        $assetData['ticket_no'] = $inputData['ticket_no'] ?? '';

        if (array_key_exists('carrier', $inputData)) {
            $carrier = Carrier::getFromName(request('carrier'))->first();
            $carrierId = $carrier ? $carrier->id : null;
            $assetData['carrier_id'] = $carrierId;
        }

        $hasRelationalValues = array_intersect_key(array_flip($relationalValues), $inputData);
        if (empty($hasRelationalValues)) {

            return $assetData;
        }

        $asset = Asset::with('assetType', 'makeAndModel', 'makeAndModel.manufacturer', 'technicalSpec')->where('serial_no', $assetData['serial_no'])->first();
        $assetData['asset_type_id'] = $assetTypeId = $asset->asset_type_id;
        $manufacturerId = $asset->makeAndModel->manufacturer->id;
        $assetData['make_and_model_id'] = $makeAndModelId = $asset->make_and_model_id;
        $assetData['technical_spec_id'] = $asset->technical_spec_id;

        if (array_key_exists('asset_type', $inputData)) {

            $assetType = AssetType::getFromName(request('asset_type'))->first();
            $assetTypeId = $assetType ? $assetType->id : null;
            $assetData['asset_type_id'] = $assetTypeId;
        }


        if (array_key_exists('hardware_standard', $inputData)) {

            $makeAndModel = MakeAndModel::getMakeModel($assetTypeId, $manufacturerId)->where('name', request('hardware_standard'))->first();
            $makeAndModelId = $makeAndModel ? $makeAndModel->id : null;
            $assetData['make_and_model_id'] = $makeAndModelId;
        }
        if (array_key_exists('technical_spec', $inputData)) {

            $technicalSpecs = TechnicalSpecs::getTechnicalSpecs($makeAndModelId)->where('details', request('technical_spec'))->first();
            $assetData['technical_spec_id'] = $technicalSpecs ? $technicalSpecs->id : null;
        }

        $errors = $this->validateRelationalFields($assetData);
        if ($errors) {
            return false;
        }


        return $assetData;
    }

    /**
     * Validate fields for asset update.
     *
     * @return \Illuminate\Support\MessageBag|null
     */
    public function validateFields()
    {
        $validator = Validator::make(
            request()->all(),
            [
                'serial_no'          => 'required|exists:assets,serial_no',
                'asset_tag'          => ['nullable', new UniqueAssetTagForSerialNo(request('serial_no'))],
                'hardware_standard'  => 'nullable|exists:make_and_models,name',
                'technical_spec'  => 'nullable|exists:technical_specs,details',
                'asset_type'      => 'nullable|exists:asset_types,name',
                'carrier'            => 'nullable|exists:carriers,name',
                'legalhold'          => 'nullable|exists:users,legalhold,0',
                'created_at'         => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'warranty_end_date'  => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'lease_start_date'   => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'lease_end_date'     => ['nullable', 'date_format:' . config('date.formats.read_date_format')],
                'lost_date'          => ['nullable', 'date_format:' . config('date.formats.read_date_format')]
            ],
            [
                'serial_no.required'          => 'The serial # is required.',
                'serial_no.exists'            => 'The serial # does not exists',
                'asset_tag.distinct' => 'The asset tag # has already been taken.',
                'asset_tag.unique' => 'The asset tag # has already been taken.',
                'hardware_standard.exists'    => 'The hardware standard does not exist.',
                'technical_spec.exists'    => 'The technical specs does not exist.',
                'asset_type.exists'    => 'The asset type does not exist.',
                'carrier.exists'    => 'The carrier does not exist.',
                'created_at.date_format' => 'The created date format is incorrect. Use ' . config('date.formats.read_date_format') . ' format',
                'lost_date.date_format' => 'The lost date format is incorrect. Use ' . config('date.formats.read_date_format') . ' format',
                'lease_end_date.date_format' => 'The lease end date format is incorrect. Use ' . config('date.formats.read_date_format') . ' format',
                'lease_start_date.date_format' => 'The lease start date format is incorrect. Use ' . config('date.formats.read_date_format') . ' format',
                'warranty_end_date.date_format' => 'The warranty end date format is incorrect. Use ' . config('date.formats.read_date_format') . ' format',

            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Validate relational fields for asset update.
     *
     * @param  array  $data
     *
     * @return \Illuminate\Support\MessageBag|null
     */
    public function validateRelationalFields($data)
    {
        $validator = Validator::make($data, [
            'make_and_model_id'  => 'required|exists:make_and_models,id',
            'technical_spec_id'  => 'required|exists:technical_specs,id',
            'asset_type_id'      => ['required', 'exists:asset_types,id', new AssetTypeValidation($data)],
        ]);

        if ($validator->fails()) {
            return $validator->errors();
        }
    }

    /**
     * Save Data from the request
     * @param  array  $assetData
     *
     * @return void
     */
    public function saveData($assetData)
    {
        $asset = Asset::where('serial_no', $assetData['serial_no'])->first();

        if ($asset) {
            $data  = $this->getAttributeData($assetData, $asset);
            $this->saveAssetHistory($asset, $data);
            $asset->update($assetData);
        }
    }

    /**
     * Save asset history.
     *
     * @param  Asset  $asset
     * @param  array  $data
     *
     * @return void
     */
    public function saveAssetHistory(Asset $asset, array $data)
    {
        $history = rtrim(trim($data['history']), ',');
        if (!$history) {
            return null;
        }
        $description = "Updated attributes of Asset <a href='" . route('assets.show', $asset->id) . "'>$asset->asset_tag</a> " . $history;

        $assetHistory = [
            'user_id' => 0,
            'asset_id' => $asset->id,
            'ticket_no' => $data['asset']['ticket_no'] ?? '',
            'ticket_service_provider' => config('ticket-integration.service'),
            'action' => 'assetAttributes_changed',
            'comments' => '',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
    }

    /**
     * Get attribute data for asset update.
     *
     * @param  array  $inputData
     * @param  Asset|null  $asset
     *
     * @return array
     */
    public function getAttributeData(array $inputData, $asset = null)
    {
        $dataToupdate['asset'] = [];
        $dataToupdate['history'] = '';

        if (isset($inputData['asset_type_id'])) {
            $dataToupdate = $this->updateValue($dataToupdate, 'asset_type_id', $inputData['asset_type_id'], $asset, 'Asset Type', 'assetType', '\App\Models\AssetType');
        }

        if (isset($inputData['make_and_model_id'])) {
            $dataToupdate = $this->updateValue($dataToupdate, 'make_and_model_id', $inputData['make_and_model_id'], $asset, 'Hardware Standard', 'makeAndModel', '\App\Models\MakeAndModel');
        }

        if (isset($inputData['technical_spec_id'])) {
            $dataToupdate = $this->updateValue($dataToupdate, 'technical_spec_id', $inputData['technical_spec_id'], $asset, 'Tech Specs', 'technicalSpec', '\App\Models\TechnicalSpecs', 'details');
        }

        if (isset($inputData['carrier_id'])) {
            $dataToupdate = $this->updateValue($dataToupdate, 'carrier_id', $inputData['carrier_id'], $asset, 'Mobile Carrier', 'carrier', '\App\Models\Carrier');
        }

        if (isset($inputData['created_at'])) {
            $dataToupdate = $this->updateDateTimeValue($dataToupdate, 'created_at', $inputData['created_at'], $asset, 'Created Date');
        }

        if (isset($inputData['warranty_end_date'])) {
            $dataToupdate = $this->updateDateValue($dataToupdate, 'warranty_end_date', $inputData['warranty_end_date'], $asset, 'Warranty End Date');
        }

        if (isset($inputData['lock_status'])) {
            $dataToupdate = $this->updateValue($dataToupdate, 'lock_status', $inputData['lock_status'], $asset, 'Lock Status');
        }

        if (isset($inputData['unlock_code'])) {
            $dataToupdate = $this->updateValue($dataToupdate, 'unlock_code', $inputData['unlock_code'], $asset, 'Unlock Code');
        }

        if (isset($inputData['lock_notes'])) {
            $dataToupdate = $this->updateValue($dataToupdate, 'lock_notes', $inputData['lock_notes'], $asset, 'Lock Note');
        }

        $customAttributes = ['make_and_model_id', 'technical_spec_id', 'carrier_id', 'asset_type_id', 'serial_no', 'created_at', 'warranty_end_date'];

        //Loop through all assets attributes
        foreach ($this->getHeaderMap() as $key => $attributeName) {
            if (in_array($key, $customAttributes)) {
                continue;
            }

            if (isset($inputData[$key])) {
                $dataToupdate = $this->updateValue($dataToupdate, $key, $inputData[$key], $asset, str_replace('(mm/dd/yyyy)', '', $attributeName), '', '');
            }
        }

        return $dataToupdate;
    }

    /**
     * Update the attribute value in data to update.
     *
     * @param  array  $dataToupdate
     * @param  string  $key
     * @param  mixed  $value
     * @param  Asset|null  $asset
     * @param  string  $attributeName
     * @param  string  $relation
     * @param  string  $relationModel
     * @param  string  $relationAttribute
     *
     * @return array
     */
    public function updateValue($dataToupdate, $key, $value, $asset, $attributeName, $relation = '', $relationModel = '', $relationAttribute = 'name')
    {
        if ($value) {
            $dataToupdate['asset'][$key] = $value;
            if ($relation) {
                $dataToupdate['history'] .= $attributeName . ' updated from ' . optional($asset->$relation)->$relationAttribute . ' to ' . $relationModel::find($value)->$relationAttribute . ', ';
            } else {

                $dataToupdate['history'] .= $attributeName . ' updated ' . ($asset->$key ? 'from ' . $asset->$key : '') . ' to ' . $value . ', ';
            }
        }

        return $dataToupdate;
    }

    /**
     * Get the header map for attributes.
     *
     * @return array
     */
    public function getHeaderMap()
    {
        return config('bulk-upload.attributeUpdateData');
    }

    /**
     * Update the array for updating the given datetime attribute of the asset.
     *
     * @param  array  $dataToupdate
     * @param  string  $key
     * @param  mixed  $value
     * @param  Asset|null  $asset
     * @param  string  $attributeName
     *
     * @return array
     */
    public function updateDateTimeValue($dataToupdate, $key, $value, $asset, $attributeName)
    {
        if ($value) {
            $dataToupdate['asset'][$key] = ($value != '') ? convert_to_db_datetime($value) : Carbon::now()->format('Y-m-d H:i:s');
            $dataToupdate['history'] .= $attributeName . ' updated ' . ($asset->$key ? 'from ' . $asset->$key : '') . ' to ' . $value . ', ';
        }

        return $dataToupdate;
    }

    /**
     * Update the array for updating the given date attribute of the asset.
     *
     * @param  array  $dataToupdate
     * @param  string  $key
     * @param  mixed  $value
     * @param  Asset|null  $asset
     * @param  string  $attributeName
     *
     * @return array
     */
    public function updateDateValue($dataToupdate, $key, $value, $asset, $attributeName)
    {
        if ($value) {
            $dataToupdate['asset'][$key] = ($value != '') ? convert_to_db_date($value) : null;
            $dataToupdate['history'] .= $attributeName . ' updated ' . ($asset->$key ? 'from ' . $asset->$key : '') . ' to ' . $value . ', ';
        }
        return $dataToupdate;
    }
}
