<?php

namespace App\Services\Api;

use App\Models\Asset;
use App\User;
use App\Models\Location;
use App\Models\AssetType;
use App\Models\Manufacturer;
use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;
use Facades\App\Repositories\BulkUpload;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;
use App\Models\AssetStatus;

class AssetCreate
{
    public function getData()
    {
        $location = Location::getFromName(request('location'))->first();
        $assetData['location_id'] = $location ? $location->id : null;

        $manufacturer = Manufacturer::getFromName(request('manufacturer'))->first();
        $manufacturerId = $manufacturer ? $manufacturer->id : null;

        $assetType = AssetType::getFromName(request('asset_type'))->first();
        $assetTypeId = $assetType ? $assetType->id : null;
        $assetData['asset_type_id'] = $assetTypeId;

        $makeAndModel = MakeAndModel::getMakeModel($assetTypeId, $manufacturerId)->where('name', request('model'))->first();
        $makeAndModelId = $makeAndModel ? $makeAndModel->id : null;
        $assetData['make_and_model_id'] = $makeAndModelId;

        $technicalSpecs = TechnicalSpecs::getTechnicalSpecs($makeAndModelId)->where('details', request('technical_spec'))->first();
        $assetData['technical_spec_id'] = $technicalSpecs ? $technicalSpecs->id : null;

        $assetData['asset_tag'] = request('asset_tag');
        $assetData['po_id'] = request('po_id');
        $assetData['serial_no'] = request('serial_no');
        $assetData['ticket_no'] = request('ticket_no');
        $assetData['carrier_id'] = request('carrier_id');
        $assetData['imei'] = request('imei');
        $assetData['asset_status_id'] = AssetStatus::getStatusID('brand_new')->first()->id;

        if (in_array(request('asset_type_id'), AssetType::leased()->pluck('id')->toArray())) {
            $assetData['lease_start_date'] = parse_date_from_db_datetime(Carbon::now());
            $assetData['lease_end_date'] = parse_date_from_db_datetime(Carbon::now()->addYears(config('date.period.leased')));
        }
        return $assetData;
    }

    public function validateFields()
    {
        $assetType = AssetType::getFromName(request('asset_type'))->first();
        $assetTypeId = $assetType ? $assetType->id : null;

        $manufacturer = Manufacturer::getFromName(request('manufacturer'))->first();
        $manufacturerId = $manufacturer ? $manufacturer->id : null;

        $makeAndModel = MakeAndModel::getMakeModel($assetTypeId, $manufacturerId)->where('name', request('model'))->first();
        $makeAndModelId = $makeAndModel ? $makeAndModel->id : null;

        $validator = Validator::make(
            request()->all(),
            [
                'location' => 'required|exists:locations,room_name',
                'asset_type' => 'required|exists:asset_types,name',
                'asset_tag' => 'required|unique:assets,asset_tag',
                'po_id' => 'required',
                'serial_no' => 'required|unique:assets,serial_no',
                // 'ticket_no' => 'required',
                'manufacturer' => 'required|exists:manufacturers,name',
                'model' => [
                    'required',
                    Rule::exists('make_and_models', 'name')->where(function ($query) use ($assetTypeId, $manufacturerId) {
                        MakeAndModel::getMakeModel($query, $assetTypeId, $manufacturerId);
                    }),
                ],
                'technical_spec' => [
                    'required',
                    Rule::exists('technical_specs', 'details')->where(function ($query) use ($makeAndModelId) {
                        TechnicalSpecs::getTechnicalSpecs($query, $makeAndModelId);
                    }),
                ],
            ],
            $messages = [
                'po_id' => 'The PO # is required.',
                'serial_no.required' => 'The serial # is required.',
                'serial_no.unique' => 'The serial # has already been taken.',
                'ticket_no.required' => 'The ticket # is required.',
                'make_and_model.required'    => 'The hardware standard is required.',
            ]
        );

        if ($validator->fails()) {
            return $validator->errors();
        }
    }
}
