<?php

namespace App\Services\Api;

use Illuminate\Http\Request;

class ApiService
{
    /**
     * Constructs success response json data
     * @param mixed $data
     * @param string $message
     * 
     * @return json
     */
    public function successResponse($data, $message = 'Success')
    {
        return response()->json([
            'success'     => true,
            'data'        => $data,
            'message'     => $message,
            'status_code' => 200,
        ]);
    }

    /**
     * Constructs error response json data
     * @param mixed $data
     * @param string $message
     * 
     * @return json
     */
    public function errorResponse($message)
    {
        return response()->json([
            'success'           => false,
            'message'       => $message,
            'status_code'   => 404,
            'data'         => [],
        ]);
    }

    /**
     * Constructs no results response json data
     * @param mixed $data
     * @param string $message
     * 
     * @return json
     */
    public function noResultResponse($message)
    {
        return response()->json([
            'success'     => true,
            'message'     => $message,
            'status_code' => 200,
            'data'        => []
        ]);
    }

    /**
     * Constructs response json data when the request results count exceedes api data limit
     * @param mixed $data
     * @param string $message
     * 
     * @return json
     */
    public function limitExceededResponse()
    {
        return response()->json([
            'success'     => false,
            'message'     => 'You cannot request more than ' . config('api.pagination_limit') . ' items',
            'status_code' => 422,
            'data'        => []
        ]);
    }

    /**
     * Constructs error response json data
     * @param mixed $data
     * @param string $message
     * 
     * @return json
     */
    public function validateErrorResponse($data, $message="Error")
    {
        return response()->json([
            'success'           => false,
            'message'       => $message,
            'status_code'   => 422,
            'data'         => $data,
        ]);
    }

    /**
     * Checks the request data limit is greater than apis data limit
     * @param mixed $limit
     * 
     * @return bool
     */
    public function checkDataLimit($limit)
    {
        if ($limit > config('api.pagination_limit')) {
            return true;
        }

        return false;
    }

    /**
     * manages the pagination options for the apis
     * @param Request $request
     * 
     * @return array
     */
    public function getPaginateOptions(Request $request)
    {
        $data = [];
        $data['offset'] = 0;
        $data['limit'] = config('api.pagination_limit');

        if ($request->has('limit')) {
            $data['limit'] = $request->limit;
        }

        if ($request->has('page')) {
            $data['offset'] = ($request->page - 1) * $data['limit'];
        }

        return $data;
    }

    /**
     * Limit the maximum number of results to 500
     * @param int $limit
     * 
     * @return [type]
     */
    public function getQueryLimit($limit = 500)
    {
        return isset($limit) ? ($limit > 500 ? 500 : $limit) : 500;
    }
}
